/*
COPYRIGHT (C) 1999  Paolo Mantegazza (mantegazza@aero.polimi.it)

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
*/


#ifndef _RTAI_NEWFIFOS_H_
#define _RTAI_NEWFIFOS_H_

#define RESET		 1
#define RESIZE		 2
#define SUSPEND_TIMED	 3
#define OPEN_SIZED	 4
#define READ_ALL_AT_ONCE 5
#define READ_TIMED	 6
#define WRITE_TIMED	 7
#define RTF_SEM_INIT	 8
#define RTF_SEM_WAIT	 9
#define RTF_SEM_TRYWAIT	10
#define RTF_SEM_TIMED_WAIT	11
#define RTF_SEM_POST	12
#define RTF_SEM_DESTROY	13
#define SET_ASYNC_SIG	14

// #ifdef MODULE
#ifdef __KERNEL__

#define RTAI_MAJOR 150

extern int rtf_init(void);

/* Attach a handler to an RT-FIFO.
 *
 * Allow function handler to be called when a user process reads or writes to 
 * the FIFO. When the function is called, it is passed the fifo number as the 
 * argument.
 */

extern int rtf_create_handler(unsigned int fifo,	/* RT-FIFO */
		int (*handler)(unsigned int fifo)	/* function to be called */);


/* Define to make compatible the call of an extended arguments handler, i.e. one
 * prototypede as "int handler(unsigned int fifo, int rw);", so that it can be
 * used with rtf_create. To install such a handler:
 * rtf_create_handler(fifo, X_FIFO_HANDLER(handler)
 */

#define X_FIFO_HANDLER(handler) ((int (*)(unsigned int))(handler))


/* Create an RT-FIFO.
 * 
 * An RT-FIFO fifo is created with initial size of size.
 * Return value: On success, size is returned. On error, -1 is returned.
 */

extern int rtf_create(unsigned int fifo, int size);


/* Reset an RT-FIFO.
 * 
 * An RT-FIFO fifo is reset by setting its buffer pointers to zero, so
 * that any existing data is discarded and the fifo started anew like at its
 * creation.
 */

extern int rtf_reset(unsigned int fifo);



/* destroy an RT-FIFO.
 * 
 * Return value: On success, 0 is returned.
 */

extern int rtf_destroy(unsigned int fifo);


/* Resize an RT-FIFO.
 * 
 * Return value: size is returned on success. On error, a negative value
 * is returned.
 */

extern int rtf_resize(unsigned int minor, int size);


/* Write to an RT-FIFO.
 *
 * Try to write count bytes to an FIFO. Returns the number of bytes written.
 */

extern int rtf_put(unsigned int fifo,	/* RT-FIFO */
		void * buf,		/* buffer address */
	       	int count		/* number of bytes to write */);



/* Read from an RT-FIFO.
 *
 * Try to read count bytes from a FIFO. Returns the number of bytes read.
 */

extern int rtf_get(unsigned int fifo,	/* RT-FIFO */
		void * buf, 		/* buffer address */
		int count		/* number of bytes to read */);


/* Open an RT-FIFO semaphore.
 *
 */

extern int rtf_sem_init(unsigned int fifo,	/* RT-FIFO */
		int value			/* initial semaphore value */);


/* Post to an RT-FIFO semaphore.
 *
 */

extern int rtf_sem_post(unsigned int fifo	/* RT-FIFO */);


/* Try to acquire an RT-FIFO semaphore.
 *
 */

extern int rtf_sem_trywait(unsigned int fifo	/* RT-FIFO */);


/* Destroy an RT-FIFO semaphore.
 *
 */

extern int rtf_sem_destroy(unsigned int fifo	/* RT-FIFO */);


/* A safe printk service, usable in real time modules.
 *
 */

extern int rt_printk(const char *fmt, ...);


/* As above but no writing to log files, just on the screen.
 *
 */

extern int rt_print_to_screen(const char *fmt, ...);


/* Just for compatibility with earlier rtai_fifos releases. No more bh and user
buffers. Fifos are now awakened immediately and buffers > 128K are vmalloced */

#define rtf_create_using_bh(fifo, size, bh_list) rtf_create(fifo, size)
#define rtf_create_using_bh_and_usr_buf(fifo, buf, size, bh_list) rtf_create(fifo, size)
#define rtf_destroy_using_usr_buf(fifo) rtf_destroy(fifo)

#else

static inline int rtf_reset(int fd)
{
	return ioctl(fd, RESET);
}

static inline int rtf_resize(int fd, int size)
{
	return ioctl(fd, RESIZE, size);
}

static inline void rtf_suspend_timed(int fd, int ms_delay)
{
	ioctl(fd, SUSPEND_TIMED, ms_delay);
}

static inline int rtf_open_sized(const char *dev, int perm, int size)
{
	int fd, err;
	if ((fd = open(dev, perm)) < 0) {
		return fd;
	}
	if ((err = ioctl(fd, RESIZE, size)) < 0) {
		return err;
	}
	return fd; 
}

static inline int rtf_read_all_at_once(int fd, void *buf, int count)
{
	struct { char *buf; int count; } args = { buf, count };
	return ioctl(fd, READ_ALL_AT_ONCE, &args);
}

static inline int rtf_read_timed(int fd, void *buf, int count, int ms_delay)
{
	struct { char *buf; int count, delay; } args = { buf, count, ms_delay };
	return ioctl(fd, READ_TIMED, &args);
}

static inline int rtf_write_timed(int fd, void *buf, int count, int ms_delay)
{
	struct { char *buf; int count, delay; } args = { buf, count, ms_delay };
	return ioctl(fd, WRITE_TIMED, &args);
}

static inline void rtf_sem_init(int fd, int value)
{
	ioctl(fd, RTF_SEM_INIT, value);
}

static inline int rtf_sem_wait(int fd)
{
	return ioctl(fd, RTF_SEM_WAIT);
}

static inline int rtf_sem_trywait(int fd)
{
	return ioctl(fd, RTF_SEM_TRYWAIT);
}

static inline int rtf_sem_timed_wait(int fd, int ms_delay)
{
	return ioctl(fd, RTF_SEM_TIMED_WAIT, ms_delay);
}

static inline void rtf_sem_post(int fd)
{
	ioctl(fd, RTF_SEM_POST);
}

static inline void rtf_sem_destroy(int fd)
{
	ioctl(fd, RTF_SEM_DESTROY);
}

static inline void rtf_set_async_sig(int fd, int signum)
{
	ioctl(fd, SET_ASYNC_SIG, signum);
}

#endif

#endif
