package org.drools.decisiontable.parser.xls;

/*
 * Copyright 2005 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.drools.template.parser.DataListener;

/**
 * Reads an Excel sheet as key-value properties.
 *
 * Treats the first non-empty cell on a row as a key and any subsequent
 * non-empty cell as a value. Any cells defined after the second cell are
 * ignored as comments.
 *
 * Could be easily adapted to accept multiple values per key but the semantics
 * were kept in line with Properties.
 *
 * @author <a href="mailto:shaun.addison@gmail.com"> Shaun Addison </a>
 *
 */
public class PropertiesSheetListener
implements
DataListener {

    private final Map<Integer,Object[]> _rowProperties = new HashMap<Integer, Object[]>();
    private final Map<String,List<String>>  _properties = new HashMap<String,List<String>>();

    /**
     * Return the key value pairs. If this is called before the sheet is
     * finished, then it will build the properties map with what is known.
     * Subsequent calls will update the properties map.
     *
     * @return properties
     */
    public List<String> getPropertyList( String key ) {
        finishSheet(); // MN allows this to be called before the sheet is
        // finished, as
        // some properties are used while the sheet is being parsed.
        return this._properties.get( key );
    }

    public List<String> getPropertyList( String key, String defaultValue ) {
        List<String> result = this.getPropertyList( key );
        if( result == null || result.size() == 0 ){
            result = new ArrayList<String>( 1 );
            result.add( defaultValue );
        }
        return result;
    }

    /*
     * (non-Javadoc)
     *
     * @see my.hssf.util.SheetListener#startSheet(java.lang.String)
     */
    public void startSheet(final String name) {
    }

    /*
     * (non-Javadoc)
     *
     * @see my.hssf.util.SheetListener#finishSheet()
     */
    public void finishSheet() {
        for ( Object[] keyValue : _rowProperties.values() ) {
            this._properties.put( (String)keyValue[0], (List<String>)keyValue[1] );
        }
        _rowProperties.clear();
    }

    /**
     * Enter a new row. This is ignored.
     *
     * @param rowNumber
     *            The row number.
     * @param columns
     *            The Colum number.
     */
    public void newRow(final int rowNumber,
            final int columns) {
        // nothing to do.
    }

    /*
     * (non-Javadoc)
     *
     * @see my.hssf.util.SheetListener#newCell(int, int, java.lang.String)
     */
    public void newCell(final int row,
            final int column,
            final String value,
            final int mergedColStart) {
        if( value == null ) return;
        String trimmedValue = value.trim();
        if( "".equals( trimmedValue ) ) return;

        final Integer rowInt = new Integer( row );
        if ( this._rowProperties.containsKey( rowInt ) ) {
            final Object[] keyValue = (Object[]) this._rowProperties.get( rowInt );
            ((List<String>)keyValue[1]).add( trimmedValue );
        } else {
            final Object[] keyValue = {trimmedValue.toLowerCase(), new ArrayList<String>()};
            this._rowProperties.put( rowInt, keyValue );
        }
    }
}

