;; Based on gradient and Step themes created by:
;; Copyright (C) 1999 John Harper <john@dcs.warwick.ac.uk>
;; Copyright (C) 1999 Michele Campeotto <micampe@f2s.com>

;; Copyright (C) 2009 Sergey Kozhemyakin <serg.kozhemyakin@gmail.com>

(require 'gradient)

(defgroup gradient-tabbed "Gradient Tabbed Theme"
  :group appearance)

(defcustom gradient-tabbed:gradient-type 'horizontal
  "Direction of gradient in `gradient' frame style."
  :type symbol
  :options (horizontal vertical diagonal)
  :group (appearance gradient-tabbed)
  :after-set after-setting-frame-option)

(defcustom gradient-tabbed:normal-from-color (get-color "#b6b6b6")
  "`From' color of inactive frames in `gradient' frame style."
  :type color
  :group (appearance gradient-tabbed)
  :after-set after-setting-frame-option)

(defcustom gradient-tabbed:normal-to-color (get-color "#323232")
  "`To' color of inactive frames in `gradient' frame style."
  :type color
  :group (appearance gradient-tabbed)
  :after-set after-setting-frame-option)

(defcustom gradient-tabbed:active-from-color (get-color "#64b4df")
  "`From' color of active frames in `gradient' frame style."
  :type color
  :group (appearance gradient-tabbed)
  :after-set after-setting-frame-option)

(defcustom gradient-tabbed:active-to-color (get-color "#000030")
  "`To' color of active frames in `gradient' frame style."
  :type color
  :group (appearance gradient-tabbed)
  :after-set after-setting-frame-option)

(defcustom gradient-tabbed:save-memory t
  "Use less memory when creating gradients, possibly affecting quality."
  :type boolean
  :group (appearance gradient-tabbed)
  :after-set after-setting-frame-option)

(let*
    ((render-bg 
      (lambda (img state)
        (apply (cond ((eq gradient-tabbed:gradient-type 'diagonal)
                      draw-diagonal-gradient)
                     ((eq gradient-tabbed:gradient-type 'horizontal)
                      draw-horizontal-gradient)
                     ((eq gradient-tabbed:gradient-type 'vertical)
                      draw-vertical-gradient))
               img (if state
                       (list gradient-tabbed:active-from-color
                             gradient-tabbed:active-to-color)
                     (list gradient-tabbed:normal-from-color
                           gradient-tabbed:normal-to-color)))
        (when (> (cdr (image-dimensions img)) 4)
          (bevel-image img 1 (not (eq state 'clicked))))
        (set-image-border img 1 1 1 1)))
     
     (scale (lambda () (if gradient-tabbed:save-memory 2 1)))
     (border-color (get-color "black"))

     (render-minimize-button 
      (lambda (img state)
        (clear-image
         img
         (if state
             gradient-tabbed:active-from-color
           gradient-tabbed:normal-from-color))
        (when (> (cdr (image-dimensions img)) 4)
          (bevel-image img 1 (not (eq state 'clicked))))
        (set-image-border img 1 1 1 1)
        ))

     (render-close-button 
      (lambda (img state)
        (clear-image
         img
         (if state
             gradient-tabbed:active-to-color
           gradient-tabbed:normal-to-color))
        (when (> (cdr (image-dimensions img)) 4)
          (bevel-image img 1 (not (eq state 'clicked))))
        (set-image-border img 1 1 1 1)
        ))

     (recolor-image (lambda (img color)
                      (when (and (imagep img) (colorp color))
                        (let*
                            ((rgb (color-rgb color))
                             (red (nth 0 rgb))
                             (green (nth 1 rgb))
                             (blue (nth 2 rgb)))
                          (set-image-modifier img 'red (list (/ red 256) 255 (/ red 256)))
                          (set-image-modifier img 'green (list (/ green 256) 255 (/ green 256)))
                          (set-image-modifier img 'blue (list (/ blue 256) 255 (/ blue 256)))))))

     (minimize-button (list
                       (recolor-image (make-image "minimize-button.png")
                                      gradient-tabbed:normal-from-color)
                       (recolor-image (make-image "minimize-button.png")
                                      gradient-tabbed:active-from-color)
                       nil
                       (make-image "minimize-pressed.png")))

     (close-button (list
                    (recolor-image (make-image "close-button.png")
                                   gradient-tabbed:normal-to-color)
                    (recolor-image (make-image "close-button.png")
                                   gradient-tabbed:active-to-color)
                    nil
                    (make-image "close-pressed.png")))

     (render-resizebar 
      (lambda (img state)
        (clear-image
         img
         (if state
             gradient-tabbed:active-from-color
           gradient-tabbed:normal-from-color))
        (when (> (cdr (image-dimensions img)) 4)
          (bevel-image img 1 (not (eq state 'clicked))))
        (set-image-border img 1 1 1 1)
        ))

     (render-resizebar-handle
      (lambda (img state)
        (clear-image
         img
         (if state
             gradient-tabbed:active-to-color
           gradient-tabbed:normal-to-color))
        (when (> (cdr (image-dimensions img)) 4)
          (bevel-image img 1 (not (eq state 'clicked))))
        (set-image-border img 1 1 1 1)
        ))

     (minimize 
      `((renderer . ,render-minimize-button)
        (foreground . ,minimize-button)
        (left-edge . 0)
        (width . 21)
        (top-edge . -21)
        (height . 21)
        (class . iconify-button)
        (removable . nil)))

     (close 
      `((renderer . ,render-close-button)
        (foreground . ,close-button)
        (right-edge . 0)
        (width . 21)
        (top-edge . -21)
        (height . 21)
        (class . close-button)
        (removable . nil)))

     (tab 
      `((renderer . ,render-bg)
        (foreground . "black")
        (top-edge . -21)
        (height . 21)
        (text . ,window-name)
        (class . tab)))

     (transient-titlebar
      `((renderer . ,render-bg)
        (foreground . "black")
        (x-justify . center)
        (y-justify . center)
        (left-edge . 0)
        (right-edge . 21)
        (top-edge . -21)
        (height . 21)
        (text . ,window-name)
        (class . title)))

     (resizebar 
      `((renderer . ,render-resizebar)
        (left-edge . 31)
        (right-edge . 31)
        (bottom-edge . -6)
        (height . 6)
        (class . bottom-border)))

     (full-resizebar-top 
      `((renderer . ,render-resizebar)
        (left-edge . 0)
        (right-edge . 0)
        (top-edge . -6)
        (height . 6)
        (class . top-border)))

     (full-resizebar-bottom 
      `((renderer . ,render-resizebar)
        (left-edge . 0)
        (right-edge . 0)
        (bottom-edge . -6)
        (height . 6)
        (class . bottom-border)))

     (resizebar-right 
      `((renderer . ,render-resizebar-handle)
        (right-edge . 0)
        (width . 30)
        (bottom-edge . -6)
        (height . 6)
        (class . bottom-right-corner)))

     (resizebar-left 
      `((renderer . ,render-resizebar-handle)
        (left-edge . 0)
        (width . 30)
        (bottom-edge . -6)
        (height . 6)
        (class . bottom-left-corner)))

     (left-border 
      `(
        (background . ,border-color)
        (right-edge . 0)
        (width . 1)
        (top-edge . 0)
        (bottom-edge . -1)
        (class . right-border)))

     (right-border 
      `(
        (background . ,border-color)
        (left-edge . 0)
        (width . 1)
        (top-edge . 0)
        (bottom-edge . -1)
        (class . left-border)))

     (bottom-border 
      `(
        (background . ,border-color)
        (left-edge . 0)
        (right-edge . 0)
        (bottom-edge . -1)
        (height . 1)
        (class . bottom-border)))

     (top-border 
      `(
        (background . ,border-color)
        (left-edge . 0)
        (right-edge . 0)
        (top-edge . -1)
        (height . 1)
        (class . top-border)))

     (frame 
      `(
        ,minimize
        ,close
        ,tab
        ,resizebar
        ,resizebar-left
        ,resizebar-right
        ))

     (shaped-frame 
      `(
        ,minimize
        ,close
        ,tab
        ))

     (transient-frame
      `(
        ,transient-titlebar
        ,close
        ,left-border
        ,right-border
        ,bottom-border
        ))

     (shaped-transient-frame
      `(
        ,transient-titlebar
        ,close
        ))
     )

  (add-frame-style 'gradient-tabbed
                   (lambda (w type)
                     (case type
                       ((default) frame)
                       ((transient) transient-frame)
                       ((shaped) shaped-frame)
                       ((shaped-transient) shaped-transient-frame))))

  (call-after-property-changed
   '(WM_NAME _NET_WM_NAME) (lambda (w prop state)
                             (reframe-window w)))

;  (custom-set-typed-variable 'tab-left-dec-width '0 'number)
;  (custom-set-typed-variable 'tab-right-dec-width '0 'number)
;  (custom-set-typed-variable 'tab-left-margin '21 'number)
;  (custom-set-typed-variable 'tab-right-margin '21 'number)
)
