/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache", "Jakarta", "JAMES" and "Apache Software Foundation"
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

package org.apache.james.mailrepository;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.james.util.Lock;
import org.apache.james.core.MailImpl;
import org.apache.mailet.Mail;
import org.apache.mailet.MailRepository;

import javax.mail.*;
import javax.mail.internet.MimeMessage;
import java.util.*;

import net.ukrpost.storage.maildir.MaildirMessage;

/** MailRepository implementation to store mail in Maildir format.
 *
 * @author Alexander Zhukov <zhukov@ukrpost.net>
 */
public class MaildirMailRepository
        extends AbstractLogEnabled
        implements MailRepository, Configurable, Serviceable, Initializable {

    /**
     * Whether 'deep debugging' is turned on.
     */
    protected final static boolean DEEP_DEBUG = false;

    private static final String TYPE = "MAIL";

    private Lock lock;
    private String destination;
    private Set keys;
    private Logger log;

    private Properties mailSessionProps = null;
    private Session mailSession = null;
    private javax.mail.Store mailStore = null;
    private Folder inbox = null;

    public void service(final ServiceManager componentManager)
            throws ServiceException {
        getLogger().debug("maildir service");
//        store = (Store)componentManager.
//        lookup( "org.apache.avalon.cornerstone.services.store.Store" );
    }

    /**
     * @see Configurable#configure(Configuration)
     */
    public void configure(Configuration conf) throws ConfigurationException {
        log = getLogger();
        log.debug("maildir configure");
        destination = conf.getAttribute("destinationURL");
        log.debug("MaildirMailRepository.destinationURL: " + destination, new Throwable());

        //this means: append "/Maildir/" to specified destination if it is not already there
        //todo: make it configurable. maybe make it javamail session property of javamaildir provider
        boolean appendMaildirIfUnavailable = true;
        if (appendMaildirIfUnavailable && !destination.endsWith("/Maildir/")) {
            if (destination.endsWith("/"))
                destination += "Maildir/";
            else
                destination += "/Maildir/";
        }

        mailSessionProps = new Properties();
        //todo: make autocreatedir configurable
        mailSessionProps.put("mail.store.maildir.autocreatedir", "true");
        mailSession = Session.getDefaultInstance(mailSessionProps);
        try {
            mailStore = mailSession.getStore(new URLName(destination));
        } catch (NoSuchProviderException e) {
            throw new ConfigurationException("cannot find store provider for " + destination, e);
        }

        String checkType = conf.getAttribute("type");
        if (!checkType.equals("MAIL")) {
            String exceptionString = "Attempt to configure MaildirMailRepository as " +
                    checkType;
            log.warn(exceptionString);
            throw new ConfigurationException(exceptionString);
        }
        log.debug("maildir configured");
    }

    /**
     * @see Initializable#initialize()
     */
    public void initialize()
            throws Exception {
        log.debug("maildir initialized");
        lock = new Lock();
        keys = Collections.synchronizedSet(new HashSet());
    }

    /**
     * Releases a lock on a message identified by a key
     *
     * @param key the key of the message to be unlocked
     *
     * @return true if successfully released the lock, false otherwise
     */
    public boolean unlock(String key) {
        log.debug("maildir unlock");
        return false;
    }

    /**
     * Obtains a lock on a message identified by a key
     *
     * @param key the key of the message to be locked
     *
     * @return true if successfully obtained the lock, false otherwise
     */
    public boolean lock(String key) {
        log.debug("maildir lock");
        return false;
    }

    /**
     * Stores a message in this repository. Shouldn't this return the key
     * under which it is stored?
     *
     * @param mc the mail message to store
     */
    public void store(Mail mc) {
        log.debug("maildir store");//, new Throwable());
        try {
            MimeMessage message = mc.getMessage();
            getInbox().appendMessages(new Message[]{message});
            log.info("message stored: " + mc.getName());
        } catch (Exception e) {
            log.error("Exception storing mail: " + e);
            e.printStackTrace();
            throw new RuntimeException("Exception caught while storing Message Container: " + e);
        }
    }

    /** Lazy-load inbox folder.
     *
     * @return inbox folder for this mail repository
     */
    private Folder getInbox() {
        if (inbox != null)
            return inbox;

        try {
            inbox = mailStore.getFolder("INBOX");
        } catch (MessagingException e) {
            throw new RuntimeException("cannot retrieve inbox for this repository", e);
        }

        return inbox;
    }

    /**
     * Retrieves a message given a key. At the moment, keys can be obtained
     * from list() in superinterface Store.Repository
     *
     * @param key the key of the message to retrieve
     * @return the mail corresponding to this key, null if none exists
     */
    public Mail retrieve(String key) {
        log.debug("maildir retrieve " + key);
        if (uniqToMessageNumber == null)
            throw new RuntimeException("uniqToMessageNumber == null");

        if (!uniqToMessageNumber.containsKey(key)) {
            log.error("no message for key " + key);
            return null;
        }

        try {
            MaildirMessage mm = getMessageFromInbox(key, openInbox());
            if (mm == null)
                return null;

            Mail mail = new MailImpl(mm);
            mail.setName(mm.getMaildirFilename().getUniq());
            return mail;
        } catch (MessagingException e) {
            e.printStackTrace();
            return null;
        } finally {
            closeInbox(false);
        }
    }

    /**
     * Removes a specified message
     *
     * @param mail the message to be removed from the repository
     */
    public void remove(Mail mail) {
        log.debug("maildir remove " + mail);
        remove(mail.getName());
    }

    /**
     * Removes a message identified by key.
     *
     * @param key the key of the message to be removed from the repository
     */
    public void remove(String key) {
        log.debug("maildir remove " + key);//, new Throwable());
        if (uniqToMessageNumber == null)
            throw new RuntimeException("uniqToMessageNumber == null");

        if (!uniqToMessageNumber.containsKey(key)) {
            log.error("no message for key " + key);
            return;
        }

        try {
            MaildirMessage mm = getMessageFromInbox(key, openInbox());
            if (mm == null)
                return;
            mm.setFlag(Flags.Flag.DELETED, true);
        } catch (MessagingException e) {
            e.printStackTrace();
        } finally {
            closeInbox(true);
        }

    }


    private Hashtable uniqToMessageNumber = new Hashtable();

    /**
     * List string keys of messages in repository.
     *
     * @return an <code>Iterator</code> over the list of keys in the repository
     *
     */
    public Iterator list() {
        log.debug("maildir list");//, new Throwable());
        Folder inbox = openInbox();
        try {
            Message mm[] = inbox.getMessages();
            if (mm == null)
                throw new RuntimeException("inbox.getMessages returned null");
            uniqToMessageNumber = new Hashtable();
            for (int i = 0; i < mm.length; i++) {
                MaildirMessage maildirMessage = (MaildirMessage) mm[i];
                Integer messageNumber = new Integer(maildirMessage.getMessageNumber());
                String uniq = maildirMessage.getMaildirFilename().getUniq();
                uniqToMessageNumber.put(uniq, messageNumber);
            }
            // Fix ConcurrentModificationException by cloning
            // the keyset before getting an iterator
            final Collection clone;
            synchronized (uniqToMessageNumber) {
                clone = new java.util.ArrayList(uniqToMessageNumber.keySet());
            }
            return clone.iterator();
        } catch (MessagingException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            closeInbox(false);
        }
    }

    private final Folder openInbox() {
        Folder inbox = getInbox();
        if (!inbox.isOpen())
            try {
                inbox.open(Folder.READ_WRITE);
            } catch (MessagingException e) {
                e.printStackTrace();
                throw new RuntimeException("cannot open inbox", e);
            }
        return inbox;
    }

    private final void closeInbox(boolean doExpunge) {
        try {
            inbox.close(doExpunge);
        } catch (MessagingException e) {
            e.printStackTrace();
        }
    }

    private final MaildirMessage getMessageFromInbox(String key, Folder inbox) throws MessagingException {
        int messageNum = ((Integer) uniqToMessageNumber.get(key)).intValue();
        MaildirMessage mm = (MaildirMessage) inbox.getMessage(messageNum);
        String uniq = mm.getMaildirFilename().getUniq();
        if (!uniq.equals(key))
            throw new RuntimeException("uniqToMessageNumber is inconsistent. '" + uniq + "' != '" + key + "'");

        return mm;
    }
}
