-- ----------
-- txid.sql
--
--	SQL script for loading the transaction ID compatible datatype 
--
--	Copyright (c) 2003-2004, PostgreSQL Global Development Group
--	Author: Jan Wieck, Afilias USA INC.
--
-- ----------

drop table txid_epoch;
drop function txid_epoch_guard();
drop type txid_snapshot cascade;
drop domain txid;

create domain txid as bigint check (value > 0);

CREATE FUNCTION get_current_txid() RETURNS bigint
	AS '$libdir/txid' LANGUAGE C;
CREATE FUNCTION get_min_txid() RETURNS bigint
	AS '$libdir/txid' LANGUAGE C;
CREATE FUNCTION get_max_txid() RETURNS bigint
	AS '$libdir/txid' LANGUAGE C;


--
-- A special transaction snapshot data type for faster visibility checks
--
CREATE FUNCTION txid_snapshot_in(cstring) RETURNS txid_snapshot
	AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;
CREATE FUNCTION txid_snapshot_out(txid_snapshot) RETURNS cstring
	AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;
CREATE FUNCTION txid_snapshot_recv(internal) RETURNS txid_snapshot
	AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;
CREATE FUNCTION txid_snapshot_send(txid_snapshot) RETURNS bytea
	AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;


--
-- The data type itself
--
CREATE TYPE txid_snapshot (
	INPUT = txid_snapshot_in,
	OUTPUT = txid_snapshot_out,
	RECEIVE = txid_snapshot_recv,
	SEND = txid_snapshot_send,
	INTERNALLENGTH = variable,
	ALIGNMENT = double
);


--
-- Special comparision functions used by the remote worker
-- for sync chunk selection
--
CREATE FUNCTION txid_in_snapshot(bigint, txid_snapshot) RETURNS boolean
	AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;

CREATE FUNCTION txid_not_in_snapshot(bigint, txid_snapshot) RETURNS boolean
	AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;

CREATE FUNCTION txid_current_snapshot() RETURNS txid_snapshot
	AS '$libdir/txid' LANGUAGE C;


CREATE FUNCTION txid_snapshot_xmin(txid_snapshot)
RETURNS bigint AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;

CREATE FUNCTION txid_snapshot_xmax(txid_snapshot)
RETURNS bigint AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;

-- prev, cur
CREATE FUNCTION txid_snapshot_old_txids(txid_snapshot, txid_snapshot)
RETURNS setof txid AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;

-- prev, cur
CREATE FUNCTION txid_array_old_txids(txid_snapshot, txid_snapshot)
RETURNS bigint[] AS '$libdir/txid' LANGUAGE C IMMUTABLE STRICT;



-- remember txid settings
-- use bigint so we can do arithmetic with it
create table txid_epoch (
	epoch bigint,
	last_value bigint
);

-- make sure there exist exactly one row
insert into txid_epoch values (0, 1);

-- then protect it
create function txid_epoch_guard() returns trigger
language plpgsql as $$
begin
    if TG_OP = 'UPDATE' then
	-- epoch: allow only small increase
	if NEW.epoch > OLD.epoch and NEW.epoch < (OLD.epoch + 3) then
	    return NEW;
	end if;
	-- last_value: allow only increase
	if NEW.epoch = OLD.epoch and NEW.last_value > OLD.last_value then
	    return NEW;
	end if;
    end if;
    raise exception 'bad operation on txid_epoch';
end;
$$;

-- the trigger
--create trigger txid_epoch_trigger
--before insert or update or delete on txid_epoch
--for each row execute procedure txid_epoch_guard();

