#!/bin/sh
#
# Copyright (c) 2024 The NetBSD Foundation, Inc.
# All rights reserved.
# 
# This code is derived from software contributed to The NetBSD Foundation
# by Nia Alarie.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#
# Goals of nbupgrade(8):
# - safety, the user should not be left with a system where the kernel
#   is older than the userspace. all checksums should be checked.
# - seamless portability, the tool should be aware of the differences
#   between NetBSD ports to different architectures, and detect things
#   like xz vs. gzip sets automatically.
# - flexibility, the tool should work on customized installations
#   regardless of which sets they have installed. It should work for
#   upgrading to different branches as well as releases.
#

UPGRADE_CACHE=/var/cache/nbupgrade
KERNEL_SETS="dtb gpufw modules"
KERNEL_VER=$(uname -v | cut -d' ' -f2) || exit 1
KERNEL_VER_MAJOR=$(uname -v | cut -d' ' -f2 | cut -d. -f1) || exit 1
KERNEL_NAME=$(sysctl -n kern.configname) || exit 1
RELEASEDIR=https://cdn.netbsd.org/pub/NetBSD
RELEASEDIR_DAILY=https://nycdn.netbsd.org/pub/NetBSD-daily
EXTRA_SETS=""

is_branch() {
	if printf "%s" "$1" | grep -q "netbsd-[[:digit:]]*$"; then
		return 0
	fi
	if [ "$1" = "head" ]; then
		return 0
	fi
	return 1
}


do_wait() {
	printf "You have five seconds to cancel with Ctrl+C.\n"
	for i in 1 2 3 4 5;
	do
		sleep 1
		printf "."
	done
	printf "\n"
}

fetch_checksum() {
	if ! [ -e "$UPGRADE_CACHE/${1}-SHA512" ]; then
		ftp -o "$UPGRADE_CACHE/${1}-SHA512" \
			"${FULL_RELEASEDIR}/binary/${1}/SHA512" || exit 1
	fi
}

fetch_set() {
	local extract_sufx="tgz"
	fetch_checksum "sets"
	if grep -q '\.tar\.xz' "${UPGRADE_CACHE}/sets-SHA512"; then
		extract_sufx="tar.xz"
	fi
	if [ -e "${UPGRADE_CACHE}/${1}.${extract_sufx}" ]; then
		return 0
	fi
	ftp -o "${UPGRADE_CACHE}/${1}.${extract_sufx}" \
		"${FULL_RELEASEDIR}/binary/sets/${1}.${extract_sufx}" || exit 1
	return 0
}

fetch_kernel() {
	if [ -e "${UPGRADE_CACHE}/${1}" ]; then
		return 0
	fi
	fetch_checksum "kernel"
	ftp -o "${UPGRADE_CACHE}/${1}" \
		"${FULL_RELEASEDIR}/binary/kernel/${1}" || exit 1
	local cksum_remote=$(grep "^SHA512 (${1})" "${UPGRADE_CACHE}/kernel-SHA512" | cut -d' ' -f4)
	local cksum_local=$(cksum -a SHA512 -q "${UPGRADE_CACHE}/${1}")
	if [ "${cksum_remote}" != "${cksum_local}" ]; then
		printf "FATAL: REMOTE AND LOCAL CHECKSUMS DIFFER!\n"
		printf "RELEASE FILE %s MAY BE CORRUPT!\n" "${UPGRADE_CACHE}/${1}\n"
		printf "Hint: delete %s or run nbupgrade -c\n" "${UPGRADE_CACHE}"
		exit 1
	fi
	return 0
}

extract_set() {
	if ! [ -e "${UPGRADE_CACHE}/sets-SHA512" ]; then
		fetch_checksum "sets"
	fi
	local filename=""
	if [ -e "${UPGRADE_CACHE}/${1}.tgz" ]; then
		filename=${1}.tgz
	elif [ -e "${UPGRADE_CACHE}/${1}.tar.xz" ]; then
		filename=${1}.tar.xz
	else
		return 1
	fi
	local cksum_remote=$(grep "^SHA512 (${filename})" "${UPGRADE_CACHE}/sets-SHA512" | cut -d' ' -f4)
	local cksum_local=$(cksum -a SHA512 -q "${UPGRADE_CACHE}/${filename}")
	if [ "${cksum_remote}" != "${cksum_local}" ]; then
		printf "FATAL: REMOTE AND LOCAL CHECKSUMS DIFFER!\n"
		printf "RELEASE FILE %s MAY BE CORRUPT!\n" \
			"${UPGRADE_CACHE}/${filename}"
		printf "Hint: delete %s or run nbupgrade -c\n" "${UPGRADE_CACHE}"
		exit 1
	fi
	progress -zf "${UPGRADE_CACHE}/${filename}" tar -C / -xpf - || return 1
	rm -f "${UPGRADE_CACHE}/${filename}" 2>/dev/null
	return 0
}

fetch_kernel_sets() {
	for set in ${KERNEL_SETS}; do
		if [ -e "/etc/mtree/set.${set}" ]; then
			fetch_set "${set}" || exit 1
		fi
	done
	fetch_set "kern-${KERNEL_NAME}" || exit 1
	if [ -e "/boot/netbsd-${KERNEL_NAME}.img" ]; then
		fetch_kernel netbsd-${KERNEL_NAME}.img.gz || exit 1
	fi
	if [ -e "/boot/netbsd-${KERNEL_NAME}.ub" ]; then
		fetch_kernel netbsd-${KERNEL_NAME}.ub.gz || exit 1
	fi
	printf "done fetching kernel sets.\n"
}

fetch_sets() {
	for set in $(ls -1 /etc/mtree | grep ^set | cut -d. -f2) ${EXTRA_SETS}; do
		fetch_set "${set}" || exit 1
	done
	printf "done fetching user sets.\n"
}

update_userspace() {
	printf "\n"
	printf "===========================\n"
	printf "Updating userspace to %s...\n" "$TARGET_VER"
	printf "===========================\n"
	printf "\n"
	fetch_sets
	for set in $(ls -1 /etc/mtree | grep ^set | cut -d. -f2) ${EXTRA_SETS}; do
		if [ "${set}" != "etc" ] && [ "${set}" != "xetc" ]; then
			printf "Installing required set %s...\n" "${set}"
			extract_set "${set}" || exit 1
		fi
	done
	if [ -e ${UPGRADE_CACHE}/etc.* ]; then
		printf "Running easy postinstall tasks...\n"
		postinstall -s ${UPGRADE_CACHE}/etc.* fix atf
		postinstall -s ${UPGRADE_CACHE}/etc.* fix blocklist
		postinstall -s ${UPGRADE_CACHE}/etc.* fix mtree
		postinstall -s ${UPGRADE_CACHE}/etc.* fix opensslcertsrehash
		postinstall -s ${UPGRADE_CACHE}/etc.* fix ptyfsoldnodes
		postinstall -s ${UPGRADE_CACHE}/etc.* fix tcpdumpchroot
	fi
}

update_kernel() {
	printf "Updating kernel to %s, will use configuration %s...\n" "$TARGET_VER" "$KERNEL_NAME"
	fetch_kernel_sets
	for set in ${KERNEL_SETS}; do
		if [ -e "/etc/mtree/set.${set}" ]; then
			printf "Installing required set %s...\n" "${set}"
			extract_set "${set}" || exit 1
		fi
	done
	printf "\n"
	printf "==================\n"
	printf "Updating kernel...\n"
	printf "==================\n"
	printf "\n"
	printf "Backing up old kernel to /onetbsd...\n"
	cp -p /netbsd /onetbsd 2>/dev/null
	printf "Extracting kernel...\n"
	extract_set "kern-${KERNEL_NAME}" || exit 1
	if [ -e "/boot/netbsd-${KERNEL_NAME}.img" ]; then
		printf "Detected RPI format kernel in /boot...\n"
		cp -p "/boot/netbsd-${KERNEL_NAME}.img" "/boot/netbsd-${KERNEL_NAME}.img.old"
		progress -ezf ${UPGRADE_CACHE}/netbsd-${KERNEL_NAME}.img.gz > \
			"/boot/netbsd-${KERNEL_NAME}.img"
	fi
	if [ -e "/boot/netbsd-${KERNEL_NAME}.ub" ]; then
		printf "Detected U-Boot format kernel in /boot...\n"
		cp -p "/boot/netbsd-${KERNEL_NAME}.ub" "/boot/netbsd-${KERNEL_NAME}.ub.old"
		progress -ezf ${UPGRADE_CACHE}/netbsd-${KERNEL_NAME}.ub.gz > \
			"/boot/netbsd-${KERNEL_NAME}.ub"
	elif [ "$(uname -m)" = "evbarm" ]; then
		printf "Warning: machine is evbarm and no U-Boot kernel is\n"
		printf "available in /boot. Check if /boot is mounted.\n"
	fi
	rm -f "${UPGRADE_CACHE}/kernel-SHA512"
	printf "\n"
	printf "=========================================================\n"
	printf "Your system\'s kernel has been replaced and new modules\n"
	printf "installed. You must reboot for this to take effect.\n\n"
	printf "Afterwards, nbupgrade should be run _again_ to update\n"
	printf "the system\'s userspace.\n\n"
	printf "Run \`# shutdown -r now\` to reboot immediately.\n"
	printf "=========================================================\n"
	printf "\n"
}

do_update() {
	local do_userspace=false
	local do_kernel=false
	if [ "$(ls -di /)" = "2 /" ]; then
		# not running in chroot
		if is_branch "$TARGET_VER"; then
			printf "Target version is a branch.\n"
			if [ -n "${TARGET_VER_MAJOR}" ] && \
			   [ "${KERNEL_VER_MAJOR}" -lt "${TARGET_VER_MAJOR}" ]; then
				printf "Kernel is new major version, will upgrade it first.\n"
				do_kernel=true
			elif [ -e /lib/libc.so ] && [ /netbsd -nt /lib/libc.so ]; then
				printf "Kernel is newer than libc, will try to upgrade userspace.\n"
				do_userspace=true
			else
				if ! [ -e /lib/libc.so ] && [ -e /bin/sh ] && \
				     [ /netbsd -nt /bin/sh ]; then
					printf "Kernel is newer than /bin/sh, will try to upgrade userspace.\n"
					do_userspace=true
				else
					printf "Kernel seems to be old, will upgrade it first.\n"
					do_kernel=true
				fi
			fi
		elif [ "$(uname -r)" = "${TARGET_VER}" ]; then
			printf "Kernel is up to date, will try to upgrade userspace to %s.\n" "$TARGET_VER"
			do_userspace=true
		else
			do_kernel=true
		fi
	else
		# running in chroot
		if [ -n "${TARGET_VER_MAJOR}" ]; then
			if [ "${KERNEL_VER_MAJOR}" -ge "${TARGET_VER_MAJOR}" ]; then
				do_userspace=true
				printf "Attempting to upgrade userspace...\n"
			else
				printf "fatal: need to upgrade kernel, can\'t in chroot\n" 2>/dev/stderr
			fi
		else
			do_userspace=true
		fi
	fi
	if $do_kernel; then
		update_kernel
	elif $do_userspace; then
		do_wait
		update_userspace
	else
		printf "Nothing to do.\n"
	fi
}

post_install() {
	local xetc_set=""
	fetch_set "etc"
	if [ -e "/etc/mtree/set.xetc" ]; then
		fetch_set "xetc"
		xetc_set="-s ${UPGRADE_CACHE}/xetc.*"
	fi
	etcupdate -al -s ${UPGRADE_CACHE}/etc.* ${xetc_set}
}

cleanup() {
	if [ -e ${UPGRADE_CACHE}/etc.* ]; then
		printf "\n"
		printf "===================================\n"
		printf "Running postinstall fix obsolete...\n"
		printf "===================================\n"
		printf "\n"
		postinstall -s ${UPGRADE_CACHE}/etc.* fix obsolete
		postinstall -s ${UPGRADE_CACHE}/etc.* fix obsolete_stand
		postinstall -s ${UPGRADE_CACHE}/etc.* fix obsolete_stand_debug
	fi
	printf "\n"
	printf "======================\n"
	printf "Removing old kernel...\n"
	printf "======================\n"
	printf "\n"
	rm -f /onetbsd 2>/dev/null
	rm -f /boot/netbsd-${KERNEL_NAME}.ub.old 2>/dev/null
	rm -f /boot/netbsd-${KERNEL_NAME}.img.old 2>/dev/null
	printf "\n"
	printf "===========================\n"
	printf "Emptying nbupgrade cache...\n"
	printf "===========================\n"
	printf "\n"
	rm -rf ${UPGRADE_CACHE:?}/*
}

if [ "$(id -u)" != 0 ]; then
	printf '%s: fatal: must run as root user; use su(1)\n' "$0" 2>/dev/stderr
	exit 1
fi

if uname -m | grep -E -q "^(evbarm|evbmips|evbsh3)"; then
	ARCH_DIR=$(uname -m)-$(uname -p)
else
	ARCH_DIR=$(uname -m)
fi

rel=${KERNEL_VER}

args=$(getopt cfknpr:s: $*)
if [ $? -ne 0 ]; then
	printf 'Usage: %s [-cfknp] [-r version] [-s set]\n' "$0"
	exit 2
fi
set -- $args

do_cleanup=false
do_fetch=false
do_kernel=false
do_post=false
do_nothing=false
while [ $# -gt 0 ]; do
	case "$1" in
		-c)
			do_cleanup=true
			;;
		-f)
			do_fetch=true
			;;
		-k)
			do_kernel=true
			;;
		-n)
			do_nothing=true
			;;
		-p)
			do_post=true
			;;
		-r)
			rel=$(printf '%s' "$2" | tr '[:upper:]' '[:lower:]')
			if [ "${rel}" = "current" ]; then
				rel="head"
			fi
			if printf '%s' "${rel}" | grep -q "_stable$"; then
				rel=$(printf '%s' "${rel}" | sed -e 's,_stable$,,g')
				rel=$(printf '%d' "${rel}" | sed -e 's/\.*//g')
				rel=$(printf 'netbsd-%d' "${rel}")
			fi
			shift
			;;
		-s)
			EXTRA_SETS="$2"
			shift
			;;
	esac
	shift
done

TARGET_VER=${rel}
if [ "${rel}" != "head" ]; then
	TARGET_VER_MAJOR=$(printf '%s' "${rel}" | sed -e 's,^netbsd-,,g' | cut -d. -f1)
fi

if [ "${TARGET_VER}" = "head" ]; then
	FULL_RELEASEDIR=$(printf "%s/HEAD/latest/%s" "$RELEASEDIR_DAILY" "$ARCH_DIR")
elif is_branch "${TARGET_VER}"; then
	FULL_RELEASEDIR=$(printf "%s/netbsd-%d/latest/%s" "$RELEASEDIR_DAILY" "$TARGET_VER_MAJOR" "$ARCH_DIR")
else
	FULL_RELEASEDIR=$(printf "%s/NetBSD-%s/%s" "$RELEASEDIR" "$TARGET_VER" "$ARCH_DIR")
fi

mkdir -p ${UPGRADE_CACHE} || exit 1

if [ -e /etc/nbupgrade.conf ]; then
	. /etc/nbupgrade.conf
fi

if $do_fetch; then
	if $do_kernel; then
		fetch_kernel_sets
	else
		fetch_kernel_sets
		fetch_sets
	fi
elif $do_kernel; then
	update_kernel
elif ! $do_nothing; then
	do_update
fi

if $do_post; then
	post_install
fi

if $do_cleanup; then
	cleanup
fi
