# See /progs/kernel/texmacs/tm_convert.scm
#     /progs/utils/edit/selections.scm

using SymPy
import Markdown

# Helpful SymPy definitions (see src/SymPy.jl for 'priviledged', etc.)
import_from(sympy, (:plot,:plot3d))
# plot3d isn't imported!
import_from(sympy, (:Sum,:Product,:fraction,:IndexedBase))
# IndexedBase isn't imported!
import_from(sympy, (:factor_terms,:logcombine,:powsimp,:separatevars,:collect,:use))


###############################################################

DATA_BEGIN   = Char(2)
DATA_END     = Char(5)
DATA_COMMAND = Char(16)
DATA_ESCAPE  = Char(27)

# Number of input line, resets every n lines
i = 1;  n = 20;
# We generalize the behavior of the REPL with an n-element array 'ans':
ans = Vector{Any}(nothing,n)


# Send the concatenation of strings s1,s2,... to TeXmacs.
function to_texmacs(s...)
  println(DATA_BEGIN, join(s), DATA_END)
end

# We remember the last n outputs in ans[1:n], and they are accessible by "ans[k]"
function prompt(k)
  println(DATA_BEGIN, "prompt#[", k, "]: ", DATA_END)
end

###############################################################

# Display a banner:
print(DATA_BEGIN, "html:")
println("<big><strong>Julia ", string(VERSION),
        "/SymPy ", string(sympy.__version__), "</strong></big><br>")
println("<p>Type ?&lt;<em>something</em>&gt; for Julia help and ??&lt;<em>something</em>&gt; for SymPy help.")
println("<p>The i-th of the last ", n, " answers is accessible by 'ans[i]'.</p>")
# The following three lines are tricky, and have to be here!!
  print(DATA_BEGIN, "verbatim:")
  prompt(1)
  print(DATA_END)
println(DATA_END)



# Main session loop.
while true
  global i, ans
  line = strip(readline())
  line != "" || continue
  print(DATA_BEGIN, "verbatim:")
  # The "verbatim" format allows the message to recursively contain
  # blocks of the same form, see below.
  try
    # doc(sin(x))
    # ?sympy[:sin]   # explicit module lookup
    # ?sympy[:powsimp]
    # doc(SymPy.mpmath[:hypercomb]) # explicit module lookup
    # doc(Poly(x^2,x), :coeffs) # coeffs is an object method of the poly instance
    # doc([x 1;1 x], :LUsolve)  # LUsolve is a matrix method
    if startswith(line, "??")  # SymPy help
      help = eval(Meta.parse(string(replace(line, "??" => "Docs.doc(sympy[:"), "])")))
      # Unsatisfactory: to_texmacs("html:", Docs.HTML(help))
      to_texmacs("verbatim:", help)
    elseif startswith(line, "?")   # Julia help
      help = eval(Meta.parse(string(replace(line, "?" => "Docs.doc("), ")")))
      # Convert markdown to html, which TeXmacs understands fully,
      # whereas it understands LaTeX only for formulas:
      to_texmacs("html:", Markdown.html(help))
    elseif endswith(line, ";")
      ans[i] = eval(Meta.parse(line))
    else
      # Parse the line into a Julia/SymPy expression, evaluate it, and set 'ans'
      # to it, like the REPL would do:
      ans[i] = eval(Meta.parse(line))
      # Decide what to do based on the type of the answer:
      typa = string(typeof(ans[i]))
      # println("%% ", typa) # diagnostic
      if startswith(typa, "Plots.Plot{")
        # The 'answer' is a Julia plot, show it. This assumes
        # "using Plots", or, better yet, "import Plots".
        Plots.gui(ans[i])
      elseif startswith(typa, "PyCall.PyObject")
        # The 'answer' is a sympy.plot(), sympy.plot3d(), etc.
        to_texmacs("verbatim: ", ans[i])
      elseif contains(typa, "Sym")
        # Non-graphics SymPy answer, convert it to LaTeX and send it to TeXmacs:
        to_texmacs("latex: \$", sympy.latex(ans[i]), "\$")
      else
        # this is a Julia answer
        # if startswith(typa, r"Int64|Float64|Array|Set|typeof")
        to_texmacs("verbatim: ", ans[i])
      end
    end
  catch err
      println("ERROR: ", err)
  end
  i = mod(i, N) + 1
  prompt(i)
  print(DATA_END)
end

#=
Open issues:
  1. Mathematical input.  See ../progs/init-julia.scm
  2. Tab completion. See tm_python. Looks like a lot of work!
#=


### Help! ###

#=
>>> help(sympy.powsimp)

class _Helper(builtins.object)
|  Define the builtin 'help'.
 This is a wrapper around pydoc.help that provides a helpful message
|  when 'help' is typed at the Python interactive prompt.
|  
|  Calling help() at the Python prompt starts an interactive help session.
|  Calling help(thing) prints help for the python object 'thing'.
| 
The SymPy documentation found at docs.sympy.org is generated from
docstrings in the source code and dedicated narrative documentation
files in the doc/src directory. Both are written in reStructuredText
format extended by Sphinx.

import docutils.core 
  
docutils.core.publish_file( 
    source_path ="restructured.rst", 
    destination_path ="Output.html", 
    writer_name ="html") 
#=
