package org.md.catalina.logger;
 
import java.io.DataInputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.net.InetAddress;
import java.net.Socket;


/** This class communicates with a SMTP server and can send an email.
 * @author cgokey@gcmd.nasa.gov
 */
public class SMTPClient {
	private Socket smtpSocket = null;
	private DataInputStream is = null;
	private PrintStream os = null;
	private String smtpHost;
	private int port = 25;

	// Constructs the object with the specified SMTP server.
	public SMTPClient(String smtpHost, int port) {
		this.smtpHost = smtpHost;
		this.port = port;
	}
	// Constructs the object with the specified SMTP server using the standard port 25.
	public SMTPClient(String smtpHost) {
		this.smtpHost = smtpHost;
	}	
	
	// Sets the SMTP server.
	public void setSmtpHost(String s) {
		this.smtpHost = s;
	}
	
	// Send an email to the specified person given the specified subject, message, and email address info.
	public void send(String fromAddress, String toAddress, String replyAddress, String subject, String message) throws SMTPException {
		String servResponse;

		if (smtpHost == null)
			exitMail("No SMTP host specified");

		try {
			smtpSocket = new Socket(smtpHost, port);
			os = new PrintStream(smtpSocket.getOutputStream());
			is = new DataInputStream(smtpSocket.getInputStream());

			servResponse = is.readLine();
			if (servResponse.startsWith("4"))
				exitMail("Connection refused by Server. Exact Message: " + servResponse);

			os.println("HELO " + InetAddress.getLocalHost().getHostName());
			servResponse = is.readLine();

			os.println("MAIL FROM: " + fromAddress);
			servResponse = is.readLine();
			if (servResponse.startsWith("5"))
				exitMail("A Server error has occurred in from address: " + servResponse);

			os.println("RCPT TO: <" + toAddress + ">");
			servResponse = is.readLine();
			if (servResponse.startsWith("5"))
				exitMail("A Server error has occurred in reply address: " + servResponse);

			os.println("DATA");
			servResponse = is.readLine();
			if (servResponse.startsWith("5"))
				exitMail("A Server error has occurred in DATA: " + servResponse);

			os.println("From: " + fromAddress);
			os.println("To: " + toAddress);

			if (subject != null) {
				os.println("Subject: " + subject);
			}

			if (replyAddress != null) {
				os.println("Reply-to: " + replyAddress);
			}

			os.println("");

			os.println(message);
			os.print("\r\n.\r\n");
			os.flush();
			servResponse = is.readLine();
			if (servResponse.startsWith("5"))
				exitMail("A Server error has occurred finishing DATA: " + servResponse);

			os.println("quit");
			servResponse = is.readLine();
			if (servResponse.startsWith("5"))
				exitMail("A Server error has occurred in QUIT: " + servResponse);
		} catch (Exception e) {
			exitMail(e.getMessage());
		} finally {
			try {
				if (is != null) is.close();
				if (os != null) os.close();
				if (smtpSocket != null) smtpSocket.close();
				is = null;
				os = null;
				smtpSocket = null;
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}
	}

	// If an exception occurs, this method will close out the connection with the
	// SMTP server and throw an exception explaining the error.	
	public void exitMail(String message) throws SMTPException {
		try {
			os.println("quit");
			if (is != null)
				is.close();
			if (os != null)
				os.close();
			if (smtpSocket != null)
				smtpSocket.close();
			is = null;
			os = null;
			smtpSocket = null;							
		} catch (Exception e) {
			e.printStackTrace();
		}
		throw new SMTPException(message);
	}
}
