package Video::TranscodeRC;

$VERSION = "0.01";

use strict;
use AnyEvent;
use File::Temp;
use IO::Socket::UNIX;
use Time::HiRes qw(time);

my $DEBUG = 1;

my %preview_commands = (
    display => 1,        
    draw    => 1,        
    fastbw  => 1,
    faster  => 1,        
    fastfw  => 1,        
    pause   => 1,        
    rotate  => 1,        
    slowbw  => 1,        
    slower  => 1,        
    slowfw  => 1,        
    toggle  => 1,        
    undo    => 1,        
);

sub get_command                 { shift->{command}                      }
sub get_ok_cb                   { shift->{ok_cb}                        }
sub get_failed_cb               { shift->{failed_cb}                    }
sub get_close_cb                { shift->{close_cb}                     }
sub get_selection_cb            { shift->{selection_cb}                 }
sub get_socket_data_cb          { shift->{socket_data_cb}               }
sub get_tc_data_cb              { shift->{tc_data_cb}                   }
sub get_socket_file             { shift->{socket_file}                  }
sub get_socket_fh               { shift->{socket_fh}                    }
sub get_socket_watcher          { shift->{socket_watcher}               }
sub get_transcode_fh            { shift->{transcode_fh}                 }
sub get_transcode_watcher       { shift->{transcode_watcher}            }
sub get_transcode_state         { shift->{transcode_state}              }
sub get_transcode_pid           { shift->{transcode_pid}                }
sub get_command_queue           { shift->{command_queue}                }

sub set_command                 { shift->{command}              = $_[1] }
sub set_ok_cb                   { shift->{ok_cb}                = $_[1] }
sub set_failed_cb               { shift->{failed_cb}            = $_[1] }
sub set_close_cb                { shift->{close_cb}             = $_[1] }
sub set_selection_cb            { shift->{selection_cb}         = $_[1] }
sub set_socket_data_cb          { shift->{socket_data_cb}       = $_[1] }
sub set_tc_data_cb              { shift->{tc_data_cb}           = $_[1] }
sub set_socket_file             { shift->{socket_file}          = $_[1] }
sub set_socket_fh               { shift->{socket_fh}            = $_[1] }
sub set_socket_watcher          { shift->{socket_watcher}       = $_[1] }
sub set_transcode_fh            { shift->{transcode_fh}         = $_[1] }
sub set_transcode_watcher       { shift->{transcode_watcher}    = $_[1] }
sub set_transcode_state         { shift->{transcode_state}      = $_[1] }
sub set_transcode_pid           { shift->{transcode_pid}        = $_[1] }
sub set_command_queue           { shift->{command_queue}        = $_[1] }

sub new {
    my $class = shift;
    my %par = @_;
    my  ($command, $ok_cb, $failed_cb, $close_cb, $selection_cb) =
    @par{'command','ok_cb','failed_cb','close_cb','selection_cb'};
    my  ($socket_data_cb, $tc_data_cb) =
    @par{'socket_data_cb','tc_data_cb'};

    my $self = bless {
        command             => $command,
        ok_cb               => $ok_cb,
        failed_cb           => $failed_cb,
        close_cb            => $close_cb,
        selection_cb        => $selection_cb,
        socket_data_cb      => $socket_data_cb,
        tc_data_cb          => $tc_data_cb,
        socket_file         => undef,
        socket_fh           => undef,
        socket_watcher      => undef,
        transcode_fh        => undef,
        transcode_watcher   => undef,
        transcode_pid       => undef,
        transcode_state     => "init",
        command_queue       => [],
    }, $class;
    
    return $self;
}

sub start {
    my $self = shift;
    
    my $command     = $self->get_command;
    my $socket_file = File::Temp::tmpnam();
    
    my $version = qx[transcode -v 2>&1];
    my $progress = $version =~ /v1\.0/ ? "--progress_off" : "--progress_meter 0";

    $command =~ s/-u\s+\d+//;
    $command .= " -J pv=cache=25 $progress --socket $socket_file -u 1";
    
    my $transcode_pid = open (my $transcode_fh, "$command 2>&1 |")
        or die "can't fork $command";
    
    my $transcode_watcher = AnyEvent->io (
        fh      => $transcode_fh,
        poll    => "er",
        cb      => sub { $self->process_transcode_output(@_) },
    );
    
    $self->set_socket_file($socket_file);
    $self->set_transcode_fh($transcode_fh);
    $self->set_transcode_watcher($transcode_watcher);
    $self->set_transcode_pid($transcode_pid);
    $self->set_transcode_state("started");
    1;
}

sub stop {
    my $self = shift;
    
    my $transcode_pid = $self->get_transcode_pid;
    kill 9, $transcode_pid;
    
    1;
}

sub process_transcode_output {
    my $self = shift;
    
    my $fh = $self->get_transcode_fh;
    my $buffer;
    
    if ( ! sysread($fh, $buffer, 4096) ) {
        close $fh;
        $self->set_transcode_fh(undef);
        $self->set_transcode_watcher(undef);
        $self->set_transcode_state("closed");
        my $close_cb = $self->get_close_cb;
        $close_cb->("tc") if $close_cb;
        return;
    }
    
    if ( $self->get_transcode_state eq 'started' and
         -e $self->get_socket_file ) {
        $self->setup_socket_watcher;
        $self->set_transcode_state("running");
    }
    
    my $tc_data_cb = $self->get_tc_data_cb;
    $tc_data_cb->($buffer) if $tc_data_cb;
    
    1;
}

sub setup_socket_watcher {
    my $self = shift;

    my $socket_fh = IO::Socket::UNIX->new (
        Type    => SOCK_STREAM,
        Peer    => $self->get_socket_file,
    ) or die "can't open socket ".$self->get_socket_file.": $!";
    
    my $socket_watcher = AnyEvent->io (
        fh      => $socket_fh,
        poll    => "er",
        cb      => sub { $self->process_socket_output(@_) },
    );
    
    $self->set_socket_fh($socket_fh);
    $self->set_socket_watcher($socket_watcher);
    
    $self->send("preview pause");
    
    1;
}

sub process_socket_output {
    my $self = shift;
    
    my $fh = $self->get_socket_fh;
    my $buffer;
    
    if ( ! sysread($fh, $buffer, 4096) ) {
        close $fh;
        $self->set_socket_fh(undef);
        $self->set_socket_watcher(undef);
        $self->stop;
        my $close_cb = $self->get_close_cb;
        $close_cb->("socket") if $close_cb;
        return;
    }

    $buffer =~ s/\s+$//;

    my $socket_data_cb = $self->get_socket_data_cb;
    $socket_data_cb->($buffer) if $socket_data_cb;

    if ( $buffer =~ /preview window close/ ) {
        $self->stop;
        return;
    }

    if ( $buffer =~ /OK/ ) {
        my $line = shift @{$self->get_command_queue};
        my $ok_cb = $self->get_ok_cb;
        $ok_cb->($line) if $ok_cb;
    }

    if ( $buffer =~ /FAILED/ ) {
        my $line = shift @{$self->get_command_queue};
        my $failed_cb = $self->get_failed_cb;
        $failed_cb->($line) if $failed_cb;
    }

    if ( @{$self->get_command_queue} ) {
        my $w;
        $w = AnyEvent->timer (
            after => 0.2,
            cb    => sub {
                $w = undef;
                $self->_send($self->get_command_queue->[0]);
            },
        );
    }
    
    1;
}

sub send {
    my $self = shift;
    my ($line) = @_;
    
    $line =~ s/\s*$//;

    push @{$self->get_command_queue}, $line;

    return if @{$self->get_command_queue} > 1;
    
    $self->_send($line);

    1;
}

sub _send {
    my $self = shift;
    my ($line) = @_;
    
    $line .= "\n";

    if ( ! syswrite($self->get_socket_fh, $line) ) {
        print "ERROR SENDING '$line'\n";
    }

    1;
}

1;

__END__

=head1 NAME

Video::TranscodeRC - Event driven API for remote controlling transcode

=head1 NOTE

This is release has nearly no documentation yet.
If you're interested in the details please contact the author.

=head1 ABSTRACT

Video::TranscodeRC privides an object oriented and event driven API
to remote control transcode using it's socket feature.

=head1 SYNOPSIS

  use Event; # or: use Glib;
  use AnyEvent;
  use Video::TranscodeRC;
  
  my $rc = Video::TranscodeRC->new (
      command     => "transcode -i $movie_file",
      close_cb => sub {
          my ($what) = @_;
          print "CLOSED: $what\n";
          $w->broadcast;
      },
      socket_data_cb => sub {
          my ($data) = @_;
          $data =~ s/\s+$//;
          print "SOCK: $data\n";
      },
      tc_data_cb => sub {
          my ($data) = @_;
          $data =~ s/\s+$//;
          $data =~ s/^/TC: /mg;
          print $data,"\n";
      },
      ok_cb => sub {
          my ($line) = @_;
          print "OK: $line\n";
      },
      failed_cb => sub {
          my ($line) = @_;
          print "FAILED: $line\n";
      },
  );

  $rc->start;

  my $watcher = AnyEvent->io (
      fh      => \*STDIN,
      poll    => "r",
      cb      => sub { control($rc) },
  );

  $w = AnyEvent->condvar;

  $w->wait;

  
=head1 DESCRIPTION

Video::TranscodeRC privides an object oriented and event driven API
to remote control transcode using it's socket feature.

This distribution ships the command line program

  transcode-rc

which offers remote controlling transcode by passing a batch file
with a bunch of commands or just by entering socket commands
to stdin.

An example batch file is shipped as batch.txt.

=head1 REQUIREMENTS

Video::TranscodeRC requires the follwing Perl modules:

  AnyEvent           >= 0.04
  Event or Glib

=head1 INSTALLATION

If your system meets the requirements mentioned above, installation
is just:

  perl Makefile.PL
  make test
  make install

=head1 AUTHORS

  Jrn Reder <joern at zyn dot de>

=head1 COPYRIGHT AND LICENSE

Copyright 2007 by Jrn Reder.

This library is free software; you can redistribute it
and/or modify it under the same terms as Perl itself.

=cut
