/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.formmodel.interactive.hypo;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.Date;

public class HypoBean {

    public static final MathContext MC = new MathContext(16, RoundingMode.HALF_UP);

    public static final BigDecimal BIG_DECIMAL_ZERO = new BigDecimal("0");
    public static final BigDecimal BIG_DECIMAL_TWO_THIRD = new BigDecimal("0.66");
    public static final BigDecimal BIG_DECIMAL_100 = new BigDecimal("100");
    public static final BigDecimal BIG_DECIMAL_12 = new BigDecimal("12");


    public static class Financing {
        private BigDecimal fLoanAmount;
        private BigDecimal fInterestRate;

        public Financing() {
            fLoanAmount = null;
            fInterestRate = null;
        }

        public BigDecimal getLoanAmount() {
            return fLoanAmount;
        }

        public void setLoanAmount(BigDecimal loanAmount) {
            fLoanAmount = HypoBean.roundAmount(loanAmount);
        }

        public BigDecimal getInterestRate() {
            return fInterestRate;
        }

        public void setInterestRate(BigDecimal interestRate) {
            fInterestRate = interestRate;
        }


        public BigDecimal getAnnualInterest() {
            if (fLoanAmount == null || fInterestRate == null) {
                return null;
            }
            BigDecimal interest = HypoBean.roundAmount(fLoanAmount.divide(HypoBean.BIG_DECIMAL_100, MC).multiply(
                    getInterestRate(), MC));
            return interest;
        }

        public BigDecimal getMonthlyInterest() {
            return calculateMonthlyAmount(getAnnualInterest());
        }

    }

    private Date fDateOfBirth;
    private BigDecimal fObjectPrice;
    private BigDecimal fObjectValue;
    private BigDecimal fEigenmittel;
    private Financing[] fFinancings = new Financing[2];
    private BigDecimal fYearlyAmortisationAmount;
    private int fAmortisationYears;
    private boolean fAmortisationCalculationByYears;
    public final static String DATEOFBIRTH = "dateOfBirth";
    public final static String OBJECTPRICE = "objectPrice";
    public final static String OBJECTVALUE = "objectValue";
    public final static String EIGENMITTEL = "eigenmittel";
    public final static String LOANAMOUNT = "loanAmount";
    public final static String MONTHLYINTEREST = "monthlyInterest";
    public final static String ANNUALINTEREST = "annualInterest";
    public final static String FINANCING = "financing";
    public final static String FINANCING1 = "financing[0]";
    public final static String FINANCING2 = "financing[1]";
    public final static String FINANCING1_LOANAMOUNT = FINANCING1 + ".loanAmount";;
    public final static String FINANCING2_LOANAMOUNT = FINANCING2 + ".loanAmount";
    public final static String FINANCING1_INTERESTRATE = FINANCING1 + ".interestRate";
    public final static String FINANCING2_INTERESTRATE = FINANCING2 + ".interestRate";
    public final static String FINANCING1_MONTHLYINTEREST = FINANCING1 + ".monthlyInterest";
    public final static String FINANCING2_MONTHLYINTEREST = FINANCING2 + ".monthlyInterest";
    public final static String FINANCING1_ANNUALINTEREST = FINANCING1 + ".annualInterest";
    public final static String FINANCING2_ANNUALINTEREST = FINANCING2 + ".annualInterest";
    public final static String YEARLYAMORTISATIONAMOUNT = "yearlyAmortisationAmount";
    public final static String AMORTISATIONAMOUNT = "amortisationAmount";
    public final static String AMORTISATIONYEARS = "amortisationYears";
    public final static String ANNUALCOSTS = "annualCosts";
    public final static String MONTHLYCOSTS = "monthlyCosts";


    public HypoBean() {
        setFinancings(0, new Financing());
        setFinancings(1, new Financing());
        setObjectPrice(null);
        setDateOfBirth(null);
        setEigenmittel(null);
        intSetYearlyAmortisationsAmount(null);
        setAmortisationYears(25);
    }


    public static BigDecimal roundAmount(BigDecimal amount) {
        if (amount == null) {
            return null;
        }
        amount.setScale(2, RoundingMode.HALF_UP);
        return amount;
    }


    public static BigDecimal roundInterest(BigDecimal interest) {
        if (interest == null) {
            return null;
        }
        interest.setScale(4, RoundingMode.HALF_UP);
        return interest;
    }


    public Date getDateOfBirth() {
        return fDateOfBirth;
    }


    public void setDateOfBirth(Date dateOfBirth) {
        fDateOfBirth = dateOfBirth;
    }

    public BigDecimal getAmortisationAmount() {
        return getFinancing(1).getLoanAmount();
    }

    public BigDecimal getObjectPrice() {
        return fObjectPrice;
    }


    public void setObjectPrice(BigDecimal objectValue) {
        fObjectPrice = roundAmount(objectValue);
        updateFinancing();
    }

    public void setFinancings(int index, Financing financings) {
        fFinancings[index] = financings;
    }

    public Financing getFinancing(int index) {
        return fFinancings[index];
    }


    public BigDecimal getEigenmittel() {
        return fEigenmittel;
    }


    public void setEigenmittel(BigDecimal eigenmittel) {
        fEigenmittel = roundAmount(eigenmittel);
        updateFinancing();
    }


    private void updateFinancing() {
        BigDecimal objectPrice = getObjectPrice();
        BigDecimal objectValue = getObjectValue();
        BigDecimal eigenmittel = getEigenmittel();
        if (eigenmittel == null) {
            eigenmittel = BIG_DECIMAL_ZERO;
        }
        if (objectValue != null && objectPrice != null) {
            BigDecimal loanAmount = objectPrice.subtract(eigenmittel, MC);
            if (loanAmount.compareTo(BIG_DECIMAL_ZERO) > 0) {
                BigDecimal firstAmount = objectValue.multiply(BIG_DECIMAL_TWO_THIRD, MC).min(loanAmount);
                updateLoanAmounts(firstAmount, loanAmount.subtract(firstAmount, MC));
            } else {
                updateLoanAmounts(BIG_DECIMAL_ZERO, BIG_DECIMAL_ZERO);
            }
        }
    }


    private void updateLoanAmounts(BigDecimal loanAmountFirst, BigDecimal loanAmountSecond) {
        getFinancing(0).setLoanAmount(loanAmountFirst);
        getFinancing(1).setLoanAmount(loanAmountSecond);
        if (fAmortisationCalculationByYears) {
            updateYearlyAmortisation();
        } else {
            updateAmortiastionYears();
        }
    }

    public BigDecimal getLoanAmount() {
        return safeAdd(getFinancing(0).getLoanAmount(), getFinancing(1).getLoanAmount());
    }


    public BigDecimal getYearlyAmortisationAmount() {
        return fYearlyAmortisationAmount;
    }


    public void setYearlyAmortisationAmount(BigDecimal yearlyAmortisationAmount) {
        intSetYearlyAmortisationsAmount(yearlyAmortisationAmount);
        fAmortisationCalculationByYears = false;
        updateAmortiastionYears();
    }


    protected void updateAmortiastionYears() {
        if (getAmortisationAmount() != null && getYearlyAmortisationAmount() != null
                && getYearlyAmortisationAmount().compareTo(BIG_DECIMAL_ZERO) > 0) {
            fAmortisationYears = getAmortisationAmount().divideToIntegralValue(getYearlyAmortisationAmount(), MC)
                    .intValue();
            if (getAmortisationAmount().compareTo(
                    getYearlyAmortisationAmount().multiply(new BigDecimal(getAmortisationYears()), MC)) > 0) {
                fAmortisationYears += 1;
            }

        } else {
            fAmortisationYears = 0;
        }
    }


    private void intSetYearlyAmortisationsAmount(BigDecimal yearlyAmortisationAmount) {
        fYearlyAmortisationAmount = roundAmount(yearlyAmortisationAmount);
    }

    public int getAmortisationYears() {
        return fAmortisationYears;
    }


    public void setAmortisationYears(int amortisationYears) {
        fAmortisationYears = amortisationYears;
        fAmortisationCalculationByYears = true;
        updateYearlyAmortisation();
    }


    private void updateYearlyAmortisation() {
        if (getAmortisationAmount() != null && getAmortisationYears() > 0) {
            intSetYearlyAmortisationsAmount(getAmortisationAmount().divide(new BigDecimal(getAmortisationYears()), MC));
        } else {
            intSetYearlyAmortisationsAmount(getAmortisationAmount());
        }
    }

    public BigDecimal getAnnualCosts() {
        return safeAdd(getYearlyAmortisationAmount(), getAnnualInterest());
    }


    public BigDecimal getAnnualInterest() {
        BigDecimal annualInterest = BIG_DECIMAL_ZERO;
        for (Financing financing : fFinancings) {
            BigDecimal interest = financing.getAnnualInterest();
            if (interest != null) {
                annualInterest = annualInterest.add(interest, MC);
            }
        }
        return annualInterest;
    }

    public BigDecimal getMonthlyInterest() {
        return calculateMonthlyAmount(getAnnualInterest());
    }

    public BigDecimal getMonthlyCosts() {
        return calculateMonthlyAmount(getAnnualCosts());

    }


    public BigDecimal getObjectValue() {
        return fObjectValue;
    }


    public void setObjectValue(BigDecimal objectValue) {
        fObjectValue = objectValue;
        updateFinancing();
    }


    private static BigDecimal safeAdd(BigDecimal amount1, BigDecimal amount2) {
        if (amount1 == null) {
            return amount2;
        }
        return amount2 == null ? amount1 : amount1.add(amount2, MC);
    }


    private static BigDecimal calculateMonthlyAmount(BigDecimal annualAmount) {
        return annualAmount == null ? null : roundAmount(annualAmount.divide(BIG_DECIMAL_12, MC));
    }

}
