# -*- coding: utf-8 -*-

"""
Font selection dialog mockup for Gtk+
Henrique C. Alves <hcarvalhoalves@gmail.com>

Most of the usability improvements of this proposal are on the Glade file,
as consistent positioning of widgets, sizes, and a more appropriate resizing
behaviour for the dialog.
"""

import gtk
import gtk.glade
import pango

GLADE_FILE = "fontselector.glade"

"""Utility methods"""

def get_size(size_widget):
    index = size_widget.get_cursor()[0][0]
    return size_widget.get_model()[index][0]


def get_size(size_widget):
    print size_widget.get_value()
    return int(size_widget.get_value())


def prepare_families(treeview):
    if treeview.get_model():
        return True
    cell = gtk.CellRendererText()
    col = gtk.TreeViewColumn(None, cell, markup=1)
    treeview.append_column(col)
    list = gtk.ListStore(pango.FontFamily, str)
    list.set_sort_column_id(1, gtk.SORT_ASCENDING)
    treeview.set_model(list)

def populate_families(treeview, parent, filter):
    prepare_families(treeview)
    list = treeview.get_model()
    list.clear()
    for family in parent.get_pango_context().list_families():
        if filter == 1:
            if family.is_monospace():
                list.append([family, family.get_name()])
        elif filter == 2:
            if not family.is_monospace():
                list.append([family, family.get_name()])
        else:
            list.append([family, family.get_name()])

def prepare_faces(treeview):
    if treeview.get_model():
        return True
    cell = gtk.CellRendererText()
    col = gtk.TreeViewColumn(None, cell, markup=1)
    treeview.append_column(col)
    list = gtk.ListStore(pango.FontFace, str)
    list.set_sort_column_id(1, gtk.SORT_DESCENDING)
    treeview.set_model(list)

def populate_faces(treeview, family, parent):
    prepare_faces(treeview)
    list = treeview.get_model()
    list.clear()
    for face in family.list_faces():
        list.append([face, face.get_face_name()])

def prepare_sizes(treeview):
    if treeview.get_model():
        return True
    cell = gtk.CellRendererText()
    col = gtk.TreeViewColumn(None, cell, markup=1)
    treeview.append_column(col)
    list = gtk.ListStore(int, str)
    treeview.set_model(list)

def populate_sizes(treeview, parent):
    prepare_sizes(treeview)
    list = treeview.get_model()
    list.clear()
    # This is a natural set of default sizes
    sizes = range(7, 14) + range(14, 28, 2) + range(28, 40, 4) + range(40, 80, 8)
    for size in sizes:
        list.append([size, str(size)])

def prepare_filter(combobox):
    if combobox.get_model():
        return True
    cell = gtk.CellRendererText()
    combobox.pack_start(cell, True)
    combobox.add_attribute(cell, 'text', 1)
    list = gtk.ListStore(int, str)
    combobox.set_model(list)
    filters = ([0, "All fonts"], [1, "Fixed-width"], [2, "Variable-width"])
    for item in filters:
        list.append(item)
    combobox.set_active(0)

"""Callbacks"""

def family_changed(treeview, treeview_faces, treeview_sizes, parent, preview):
    index = treeview.get_cursor()[0][0]
    family = treeview.get_model()[index][0]
    populate_faces(treeview_faces, family, parent)
    treeview_faces.set_cursor(0) # Keep something selected, please!
    preview_font(preview, pango_family=family)

def face_changed(treeview, parent, preview):
    index = treeview.get_cursor()[0][0]
    face = treeview.get_model()[index][0]
    preview_font(preview, pango_face=face)

def size_changed(treeview, entry, preview):
    size = get_size(treeview)
    entry.set_text(str(int(size)))
    preview_font(preview, pango_size=size)

def size_entry_changed(entry, preview):
    size = int(str(entry.get_text()))
    if not size:
        return
    preview_font(preview, pango_size=size)

def filter_changed(combobox, treeview_family, parent):
    index = combobox.get_active()
    filter = combobox.get_model()[index][0]
    populate_families(treeview_family, parent, filter)


def preview_font(preview, pango_family=None, pango_face=None, pango_size=None):
    """
    I'm using the font's own name as the preview text here, which is not the
    best solution.

    We should use pango_language_get_sample_string() [1], as it's adequate to
    render the selected font's glyphs as pointed by Behdad [2].

    Unfortunately, PyGTK+ bindings don't have it yet =(

    [1] http://bugzilla.gnome.org/show_bug.cgi?id=473806
    [2] http://mail.gnome.org/archives/gtk-devel-list/2008-August/msg00149.html
    """

    buffer = preview.get_buffer()
    fontdesc = preview.get_pango_context().get_font_description()
    old_text = buffer.get_text(buffer.get_start_iter(), buffer.get_end_iter())
    if old_text == fontdesc.to_string():
        text_set = False
    else:
        text_set = True
    if pango_family:
        fontdesc.set_family(pango_family.get_name())
    if pango_face:
        fontdesc.merge(pango_face.describe(), True)
    if pango_size:
        size = pango_size * pango.SCALE
        fontdesc.set_size(size)
    preview.modify_font(fontdesc)
    if not old_text or not text_set:
        buffer.set_text(fontdesc.to_string())


"""Main"""

if __name__ == '__main__':
    tree = gtk.glade.XML(GLADE_FILE, "fontdialog")
    fontdialog = tree.get_widget("fontdialog")
    fontfamily = tree.get_widget("fontfamily")
    fontface = tree.get_widget("fontface")
    font_scale = tree.get_widget("font_scale")
    fontsizeentry = tree.get_widget("fontsizeentry")
    fontfilter = tree.get_widget("fontfilter")
    preview = tree.get_widget("fontpreview")

    # Callbacks
    fontfamily.connect("cursor-changed",
            family_changed, fontface, font_scale, fontdialog, preview)
    fontface.connect("cursor-changed",
            face_changed, fontdialog, preview)
    font_scale.connect("value-changed",
            size_changed, fontsizeentry, preview)
    fontsizeentry.connect("activate",
            size_entry_changed, preview)
    fontfilter.connect("changed",
            filter_changed, fontfamily, fontdialog)

    # Initialization
    prepare_filter(fontfilter)
    populate_families(fontfamily, fontdialog, 0)
#    populate_sizes(fontsize, fontdialog)

    # Select some reasonable defaults
    fontfamily.set_cursor(0)
    fontface.set_cursor(0)
    print dir(font_scale)
#    fontsize.set_cursor(5)
    size_changed(font_scale, fontsizeentry, preview)

    fontdialog.run()
