/* =============================================================================
 * Author   : Axxerion B.V.
 * Copyright: (c) 2012 Axxerion B.V., the Netherlands. All rights reserved.
 * =============================================================================
 * No part of this file may be reproduced or transmitted in any form or by any
 * means, electronic or mechanical, for the purpose, without the express written
 * permission of the copyright holder.
 * =============================================================================
 */

package com.acme.karaf.commands;


import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.karaf.bundle.core.BundleService;
import org.apache.karaf.shell.api.action.Action;
import org.apache.karaf.shell.api.action.Argument;
import org.apache.karaf.shell.api.action.Command;
import org.apache.karaf.shell.api.action.lifecycle.Reference;
import org.apache.karaf.shell.api.action.lifecycle.Service;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

@Service
@Command(scope = "acme", name = "deps", description = "Show dependencies on a module")
public class RevTreeShow implements Action {

    @Argument(index = 0, name = "id", description = "The bundle ID or name or name/version", required = true, multiValued = false)
    String id;

    @Reference
    BundleContext bundleContext;

    @Reference
    BundleService bundleService;

    public Object execute() throws Exception {
        Bundle bundle = bundleService.getBundle(id);
        return doExecute(bundle);
    }


    public Object doExecute(Bundle bundle) {
        System.out.println();
        printHeader(bundle);
        System.out.println();

        List<Bundle> allBundles = bundleService.selectBundles(Collections.emptyList(), true);
        int namelen = allBundles.stream().map(Bundle::getSymbolicName).mapToInt(String::length).max().orElse(1);

        {
            System.out.println("REQUIRED");
            System.out.println();
            Map<String, Bundle> wires = bundleService.getWiredBundles(bundle);
            for (Map.Entry<String, Bundle> wire : wires.entrySet()) {
                Bundle otherBundle = wire.getValue();
                System.out.println(
                    String.format("%-" + namelen + "s <- [%-3d] %-" + namelen + "s",
                        wire.getKey(),
                        otherBundle.getBundleId(),
                        otherBundle.getSymbolicName()));
            }
        }

        System.out.println();
        System.out.println("PROVIDED");
        System.out.println();


        for (Bundle b : allBundles) {
            Map<String, Bundle> wires = bundleService.getWiredBundles(b);
            for (Map.Entry<String, Bundle> wire : wires.entrySet()) {
                Bundle otherBundle = wire.getValue();
                if (otherBundle.getBundleId() == bundle.getBundleId()) {
                    System.out.println(
                        String.format("%-" + namelen + "s -> [%-3d] %-" + namelen + "s",
                            wire.getKey(),
                            b.getBundleId(),
                            b.getSymbolicName()));
                }
            }
        }

        System.out.println();

        return null;
    }

    private void printHeader(Bundle bundle) {
        System.out.printf("Bundle %s [%s] is currently %s%n",
            bundle.getSymbolicName(),
            bundle.getBundleId(),
            getState(bundle));
    }

    private String getState(Bundle bundle) {
        switch (bundle.getState()) {
            case Bundle.UNINSTALLED:
                return "UNINSTALLED";
            case Bundle.INSTALLED:
                return "INSTALLED";
            case Bundle.RESOLVED:
                return "RESOLVED";
            case Bundle.STARTING:
                return "STARTING";
            case Bundle.STOPPING:
                return "STOPPING";
            case Bundle.ACTIVE:
                return "ACTIVE";
            default:
                return "UNKNOWN";
        }
    }
}
