package artemis.simple.producer;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.qpid.jms.JmsConnectionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.jms.Connection;
import jakarta.jms.JMSException;
import jakarta.jms.Message;
import jakarta.jms.MessageProducer;
import jakarta.jms.Session;

public class SimpleProducer {

    private static final String MESSAGE_ID_PROPERTY = "messageId";
    private static final String MESSAGE_CONTENT_PROPERTY = "content";

    private static final Logger LOG = LoggerFactory.getLogger(SimpleProducer.class);

    private static final int PORT_BASE = 5670;
    private static final String PORT_INCREMENT_PROPERTY = "portIncrement";
    private static final String topicName = "topic";

    private final ScheduledExecutorService scheduledExecutor = Executors
            .newSingleThreadScheduledExecutor();

    private int port;
    private final AtomicInteger messageCounter = new AtomicInteger(0);

    public void startProducer() throws JMSException, JMSException {
        port = PORT_BASE + Integer.valueOf(System.getProperty(PORT_INCREMENT_PROPERTY));
        LOG.info("Starting producer on port {}", port);

        final JmsConnectionFactory jmsConnectionFactory = new JmsConnectionFactory(
                "amqp://localhost:" + port);
        final Connection connection = jmsConnectionFactory.createConnection();

        final Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        final MessageProducer producer = session.createProducer(session.createTopic(topicName));

        scheduledExecutor.scheduleWithFixedDelay(() -> publishMessage(session, producer), 10000, 1,
                TimeUnit.MILLISECONDS);
    }

    private void publishMessage(final Session session, final MessageProducer producer) {
        try {
            final Message message = session.createMapMessage();
            message.setStringProperty(MESSAGE_CONTENT_PROPERTY, "Msg to port " + port);
            message.setIntProperty(MESSAGE_ID_PROPERTY, messageCounter.incrementAndGet());

            producer.send(message);

        } catch (final Exception e) {
            LOG.error("Error during publishing of message: {}", e.toString(), e);
        }
    }

    public static void main(final String[] args) throws JMSException {
        final SimpleProducer p = new SimpleProducer();
        p.startProducer();
    }
}
