! libcube.f90

! Version: 0.0.0, 2014/02/08

! Copyrights: None

! Contributors: Created by yhli

! This file defines the following subroutines that are used to manipulate Gauss-
! i98 cube files:
!
! (1) cube_read_natom
!     This subroutine reads number of atoms from cube file.
!
! (2) cube_read_meshinfo
!     This subroutine reads the position of (0,0,0) point, size and dx along ea-
!     ch direction of the grid from cube file.
!
! (3) cube_read_atominfo
!     This subroutine reads atom Zval, charge and position from cube file.
!
! (4) cube_read_scafield
! 	  This subroutine reads scalar field from cube file.
!
! (5) cube_write
!     This subroutine writes cube file.
!
! For their interfaces, see the definitions below.


subroutine cube_read_natom( filename, natom )

    implicit none
    
    ! subroutine interface
    character(len=32), intent(in)  :: filename
    integer(kind=4),   intent(out) :: natom
    
    ! local variables
    character(len=32), dimension(2) :: header
    
    ! open cube file
    open ( unit=8, file=filename, status="old" )
    
    ! read header
    read (8,*) header(1)
    read (8,*) header(2)
    
    ! read natom
    read (8,*) natom
    
    ! close cube file
    close ( unit=8 )
    
end subroutine cube_read_natom


subroutine cube_read_meshinfo( filename, mesh_origin, mesh_size, mesh_unit )

    implicit none
    
    ! subroutine interface
    character(len=32),               intent(in)  :: filename
    real(kind=8),    dimension(3),   intent(out) :: mesh_origin
    integer(kind=4), dimension(3),   intent(out) :: mesh_size
    real(kind=8),    dimension(3,3), intent(out) :: mesh_unit
    
    ! local variables
    character(len=32), dimension(2) :: header
    integer(kind=4) :: natom
    ! loop counters
    integer(kind=4) :: i
    
    ! open cube file
    open ( unit=8, file=filename, status="old" )
    
    ! read header
    read (8,*) header(1)
    read (8,*) header(2)
    
    ! read natom and mesh_origin
    read (8,*) natom, mesh_origin(:)
    
    ! read mesh_size and mesh_unit
    do i = 1, 3
        read (8,*) mesh_size(i), mesh_unit(i,:)
    end do
    
    ! close cube file
    close ( unit=8 )
    
end subroutine cube_read_meshinfo


subroutine cube_read_atominfo( filename, natom, atom_number, atom_charge, atom_pos )

    implicit none
    
    ! subroutine interface
    character(len=32),                   intent(in)  :: filename
    integer(kind=4),                     intent(in)  :: natom
    integer(kind=4), dimension(natom),   intent(out) :: atom_number
    real(kind=8),    dimension(natom),   intent(out) :: atom_charge
    real(kind=8),    dimension(natom,3), intent(out) :: atom_pos
    
    ! local variables
    character(len=32), dimension(2) :: header
    integer(kind=4) :: shader_natom
    real(kind=8), dimension(3) :: mesh_origin
    integer(kind=4), dimension(3) :: mesh_size
    real(kind=8), dimension(3,3) :: mesh_unit
    ! loop counters
    integer(kind=4) :: i
    
    ! open cube file
    open ( unit=8, file=filename, status="old" )
    
    ! read header
    read (8,*) header(1)
    read (8,*) header(2)
    
    ! read shader_natom and mesh_origin
    read (8,*) shader_natom, mesh_origin(:)
    
    ! read mesh_size and mesh_unit
    do i = 1, 3
        read (8,*) mesh_size(i), mesh_unit(i,:)
    end do
    
    ! read atominfo
    do i = 1, natom
        read (8,*) atom_number(i), atom_charge(i), atom_pos(i,:)
    end do
    
    ! close cube file
    close ( unit=8 )
    
end subroutine cube_read_atominfo


subroutine cube_read_scafield( filename, mesh_size, scafield )

    implicit none
    
    ! subroutine interface
    character(len=32),                                                  intent(in)  :: filename
    integer(kind=4), dimension(3),                                      intent(in)  :: mesh_size
    real(kind=8),    dimension(mesh_size(1),mesh_size(2),mesh_size(3)), intent(out) :: scafield
    
    ! local variables
    character(len=32), dimension(2)   :: header
    integer(kind=4)                   :: natom
    real(kind=8),      dimension(3)   :: mesh_origin
    integer(kind=4),   dimension(3)   :: shader_mesh_size
    real(kind=8),      dimension(3,3) :: mesh_unit
    integer(kind=4)                   :: atom_number_i
    real(kind=8)                      :: atom_charge_i
    real(kind=8),      dimension(3)   :: atom_pos_i
    ! loop counters
    integer(kind=4) :: i, j
    
    ! open cube file
    open ( unit=8, file=filename, status="old" )
    
    ! read header
    read (8,*) header(1)
    read (8,*) header(2)
    
    ! read natom and mesh_origin
    read (8,*) natom, mesh_origin(:)
    
    ! read shader_mesh_size and mesh_unit
    do i = 1, 3
        read (8,*) shader_mesh_size(i), mesh_unit(i,:)
    end do
    
    ! read atominfo
    do i = 1, natom
        read (8,*) atom_number_i, atom_charge_i, atom_pos_i(:)
    end do
    
    ! read scafield
    do i = 1, mesh_size(1)
        do j = 1, mesh_size(2)
            read (8,*) scafield(i,j,:)
        end do
    end do
    
    ! close cube file
    close ( unit=8 )
    
end subroutine cube_read_scafield


subroutine cube_write( filename, header, natom, mesh_origin, mesh_size, mesh_unit, &
                       atom_number, atom_charge, atom_pos, scafield )

    implicit none
    
    ! subroutine I/O
    character(len=32),                     intent(in) :: filename
    character(len=32), dimension(2),       intent(in) :: header
    integer(kind=4),                       intent(in) :: natom
    real(kind=8),      dimension(3),       intent(in) :: mesh_origin
    integer(kind=4),   dimension(3),       intent(in) :: mesh_size
    real(kind=8),      dimension(3,3),     intent(in) :: mesh_unit
    integer(kind=4),   dimension(natom),   intent(in) :: atom_number
    real(kind=8),      dimension(natom),   intent(in) :: atom_charge
    real(kind=8),      dimension(natom,3), intent(in) :: atom_pos
    real(kind=8),      dimension(mesh_size(1),mesh_size(2),mesh_size(3)), intent(in) :: scafield
    
    ! loop counters
    integer(kind=4) :: i, j, k
    
    ! open cube file
    open ( unit=8, file=filename, status="replace" )
    
    ! write header
    write (8,*) header(1)
    write (8,*) header(2)
    
    ! write natom and mesh origin
    write (8,"(I5,3F12.6)") natom, mesh_origin(:)
    
    ! write mesh size and mesh unit
    do i = 1,3
        write (8,"(I5,3F12.6)") mesh_size(i), mesh_unit(i,:)
    end do
    
    ! write atoms info
    do i = 1,natom
        write (8,"(I5,4F12.6)") atom_number(i), atom_charge(i), atom_pos(i,:)
    end do
    
    ! write field value
    do i = 1, mesh_size(1)
        do j = 1, mesh_size(2)
            do k = 1, mesh_size(3)
                write(8,"(ES13.5)",advance="no") scafield(i,j,k)
                if (mod(k,6)==0.or.k==mesh_size(3)) then
                    write(8,*)
                end if
            end do
        end do
    end do
    
    ! close cube file
    close ( unit=8 )

end subroutine cube_write
