#include "boost/test/unit_test.hpp"
#include "boost/test/test_tools.hpp"

#include <string>
#include <exception>
#include <stdexcept>
#include <iostream>

#include <stdint.h>
#include <unistd.h>

#include "qpid/types/Exception.h"
#include "qpid/messaging/Sender.h"
#include "qpid/messaging/Message.h"
#include "qpid/messaging/Session.h"
#include "qpid/messaging/Message.h"
#include "qpid/messaging/Receiver.h"
#include "qpid/messaging/Duration.h"
#include "qpid/messaging/Connection.h"

#include "QpidBroker.h"

#include <boost/lexical_cast.hpp>

using namespace qpid::messaging;

BOOST_AUTO_TEST_SUITE(Messaging_testSuite)
 
BOOST_AUTO_TEST_CASE(Sender_ttl_testcase_1)
{
   // Demonstrate, that when several messages, with an associated TTL setting, are sent by an 
   // unconnected sender, they are placed in the outgoing queue, and still get sent on a successful 
   // reconnect, even if the time to reconnect to the Broker exceeds the message TTL. 
   try {
     QpidBroker broker("");
     broker.start(5);
   
     qpid::messaging::Connection connection("localhost","");
     connection.open();
     BOOST_CHECK(connection.isOpen());
   
     qpid::messaging::Session  session  = connection.createSession();
     qpid::messaging::Sender   sender   = session.createSender("amq.topic/fred");
     qpid::messaging::Receiver receiver = session.createReceiver("amq.topic/fred");
   
     sender.setCapacity(100);
     receiver.setCapacity(10);
   
     // Confirm Sender and receiver are working.
     const std::string content("Sender_ttl_testcase_1");
     qpid::messaging::Message sendMessage;
     sendMessage.setTtl(qpid::messaging::Duration(1000));
     sendMessage.setContent(content);
     sender.send(sendMessage,true);
   
     qpid::messaging::Message receiveMessage;
     if(!receiver.get(receiveMessage,qpid::messaging::Duration::SECOND)){
        throw std::runtime_error("Sender_ttl_testcase_1 failed : did not receive test message");
     }
  
     BOOST_CHECK(receiveMessage.getContent() == content);
     
     session.acknowledge(receiveMessage,true);
     session.sync(true);
     
     // Stop the broker to simulate a broker failure,
     // while sender and receiver 
     broker.stop(2);
     BOOST_CHECK(connection.isOpen() == false);
     
     // Send messages, but do not send more than 
     // capacity/4 as this causes a flush in the
     // sender implementaion.
     for(uint32_t x=0; x < 10; ++x){
        qpid::messaging::Message message;
        message.setTtl(qpid::messaging::Duration(1000));
        message.setContent(content + " : " + boost::lexical_cast<std::string>(x)); 
        try {
	  sender.send(message,false);
        }
        catch(const qpid::types::Exception& qme){

        }
      }  
      
      // pause the process for a time greater than the TTL
      // of the messagings in the disconnected sender queue. 
      sleep(2);
      broker.start(2);
      
      connection.reconnect();
      BOOST_CHECK(connection.isOpen());
      BOOST_CHECK(session.hasError() == false);
      
      uint32_t receiveCount = 0;
      qpid::messaging::Message receiveMessage2;
      while(receiver.get(receiveMessage2,qpid::messaging::Duration::SECOND)){
	 ++receiveCount;
	 session.acknowledge(receiveMessage,false);
      }

      session.sync(true);

     
      // The TTL on the sent messages should have expired so no
      // messages should have been received. 
      std::cout << "receiveCount : " << receiveCount << std::endl;
      BOOST_CHECK(receiveCount == 0);
      broker.stop(2);
   }
   catch(const qpid::types::Exception& qme){
      BOOST_FAIL(std::string("Sender_ttl_testcase_1 failed with qpid exception : ") + qme.what());
   }
   catch(const std::exception& e){
      BOOST_FAIL(std::string("Sender_ttl_testcase_1 failed with c++ exception : ") + e.what());
   }
}
 
BOOST_AUTO_TEST_CASE(Receiver_ttl_testcase_2)
{
   // Demonstrate that when several messages with an associated TTL setting are sent by 
   // a sender, that the messages cached by the associated receiver (capacity), for a 
   // time period greater than the messages TTL, are still delivered to the application 
   // even though the TTL has expired.The messages that are not cached are removed from 
   // the receivers broker queue.
   
   try {
     QpidBroker broker("");
     broker.start(5);
   
     qpid::messaging::Connection connection("localhost","");
     connection.open();
     BOOST_CHECK(connection.isOpen());
   
     qpid::messaging::Session  session  = connection.createSession();
     qpid::messaging::Sender   sender   = session.createSender("amq.topic/fred");
     qpid::messaging::Receiver receiver = session.createReceiver("amq.topic/fred");
   
     receiver.setCapacity(10);

     const std::string content("Receiver_ttl_testcase_2");
     for(uint32_t x=0; x < 100; ++x){
        qpid::messaging::Message message;
        message.setTtl(qpid::messaging::Duration(1000));
        message.setContent(content + " : " + boost::lexical_cast<std::string>(x)); 
	sender.send(message,false);
     }  
     
     session.sync(true);
     
     // Allow the messages in the receivers broker queue to expire
     sleep(2);
     
     // No messages should get delivered to the application because 
     // their TTL expired.
      uint32_t receiveCount = 0;
      for(uint32_t x=0; x < 3; ++x){
         qpid::messaging::Message receiveMessage;
         while(receiver.get(receiveMessage,qpid::messaging::Duration::SECOND)){
	    ++receiveCount;
	 }
      }   

      // The application will get the messages cached by
      // the receivers queue, this is sized by the capacity.
      std::cout << "receiveCount : " << receiveCount << std::endl;
      BOOST_CHECK(receiveCount == 0);
      broker.stop(2);
   }
   catch(const qpid::types::Exception& qme){
      BOOST_FAIL(std::string("Receiver_ttl_testcase_2 failed with qpid exception : ") + qme.what());
   }
   catch(const std::exception& e){
      BOOST_FAIL(std::string("Receiver_ttl_testcase_2 failed with c++ exception : ") + e.what());
   } 
}

BOOST_AUTO_TEST_SUITE_END()
 