""" The Even Simpler Simple API for XML (ESSAX) is intended to be exactly what
    the name implies, an exceptionally easy mechanism for Python programs to 
    read XML documents. Essax is designed with ease of use as the primary 
    design consideration. A summary of features for Essax includes:
        
        * Subclasses of EssaxParser need only define a method named 
          'tag_<tagname>(self)' to support an XML tag.                  
        
        * Built-in DTD-like validation
            - While not as comprehensive as a real DTD, Essax's validation
              capabilities should be sufficient for most needs and does not
              require the use of a validating XML parser.
              
        * Automatic validation, type-checking, and type-conversion of 
          attributes
            
        * Variations & Extentions of XML documents can be handled through
          inheritance/mix-ins. 
          
        * If any errors are encoutered during the parsing process,
          the file-name and line number will be reported along with
          a descriptive message.
          
    The primary niche Essax is designed to fill is where extreme effeciency is
    of less importance than ease of implementation (configuration files,
    moderately sized data files, etc). 
"""

from   collections import deque

import xml.sax
import xml.sax.handler
from   xml.sax.xmlreader import XMLReader
from   xml.sax.saxutils  import XMLFilterBase




class RuntimeParseError (Exception):
    """ This is the base class for all exceptions thrown by the parser at
    run time. If any exceptions are thrown by user code, an instance of this
    class will be inserted into the user exception's dictionary as 
    essax_exception. This will allow user code to easily detect the file and 
    line number the parser was working on when the failure occured."""
    
    def __init__(self, parser):
        self.parser      = parser
        self.file_name   = parser.file_name
        self.line_number = parser.this_tag.line_number
                

        
class MissingRequiredAttribute (RuntimeParseError):
    """ Instances of this class are thrown if a tag is encoutered that does
    not contain an attribute the user specified as required. """
    
    
    def __init__(self, tag_name, attr_name, parser):
        RuntimeParseError.__init__(self, parser)
        self.tag_name  = tag_name
        self.attr_name = attr_name
        
    def __str__(self):
        msg = 'Missing Required Attribute "%s" in tag "%s"' %\
              (self.attr_name, self.tag_name)
        return msg
               
    

class AttributeConversionError (RuntimeParseError):
    """ If the user specified a type for an attribute and the type conversion
    method cannot successfully convert the attribute text to the correct type
    (such as if the type was supposed to be an integer and the attribute text
    was 'foo'), AttributeConversionError will be raised. """
    
    def __init__(self, parser, message):
        RuntimeParseError.__init__(self, parser)
        self.message = message
        
    def __str__(self):
        return 'Attribute Conversion Error: ' + self.message


        
class UnknownTag (RuntimeParseError):
    """ This exception is raised if a tag is encountered for which there is
    no corresponding 'tag_<tagname>()' method or 'tag_begin_<tagname>()'
    /'tag_end_<tagname>()' method pair. This exception is only raised if
    exception generation on unknown tags is requested. The default is to 
    silently ignore unknown tags."""
    
    def __init__(self, tag_name, parser):
        RuntimeParseError.__init__(self, parser)
        
        self.tag_name = tag_name
        
    def __str__(self):
        return 'Unknown XML Tag "%s"' % self.tag_name


    
    
#------------------------------------------------------------------------------
# Validation Constants & Exceptions
#
V_OK         = 1
V_DONE       = 2
V_CHECK_NEXT = 3


class InvalidTagFormat (Exception):
    """ This is raised if the tag format the user registered is invalid/
    fails to properly parse (i.e the developer screwed up). 
    It is not a runtime error. """
    pass


class ValidationError (RuntimeParseError):
    """ If the user specified a DTD-like description for the valid contents
    of an XML tag and the parser determines that the current XML document
    does not match the description, an instance of this class will be raised.
    The detail of the generated error message will depend on the nature of the
    error. The current validator can describe some errors better than others.
    """
    
    def __init__(self, msg):
        self.msg = msg
    
    def set_parser(self, parser):
        RuntimeParseError.__init__(self, parser)
        
    def __str__(self):
        return self.msg
#------------------------------------------------------------------------------    
    
    



#------------------------------------------------------------------------------
# GenericTag
#
#
class GenericTag:
    """Note: Instances of this class should be created only by EssaxParser
    
    Member variables:
    
    name        - Name of the Tag this element represents
    line_number - The line on which the tag starts
    attrs       - A dictionary of the attributes. If the user specified type
                  information for the attributes, the values will be instances
                  of the specified type.
    
    *** The following variables will not contain useful data in a 
    *** tag_begin_<tagname> call since the information isn't yet available
    
    text     - The textual data contained between <tagname> </tagname> pairs
    
    sub_tags - A list of GenericTags instances for each tag under this one.
    """
    def get_text(self):
        if isinstance(self._text, list):
            if len(self._text) == 0:
                # For no text and/or during 'tag_begin_' calls
                return ''
            else:
                # For having text during 'tag_' and 'tag_end_' calls
                self._text = ''.join(self._text)
                
        return self._text

    
    text = property(get_text, None, None)
    
    
    def __init__(self, name, line_number):
        
        self.name          = name
        self.line_number   = line_number        
        self.sub_tags      = list()
        self.attrs         = None
        self.keep_subtags  = False
        
        self._text         = list() # list of string contents
        self._validator    = None   # Used directly by EssaxParser
                                    
        

        
#------------------------------------------------------------------------------
# Essax Parser
#
class EssaxParser (XMLFilterBase):
    """ This is the abstract ESSAX base class. It derives from the SAX class
    XMLFilterBase and is used to further simplify the SAX API. Most of the
    common XML parsing operations are directly supported by Essax. However,
    in the event that the Essax API does not provide the needed functionality,
    subclasses of EssaxParser are, of course, free to access the SAX API
    directly.
    
    Classes deriving from EssaxParser may support tags in one of two methods.
    Either the subclass may define a single tag_<tagname> 
    method which will be called when the closing </tagname> identifier 
    is encountered OR the subclass may define a pair of methods,
    named tag_begin_<tagname> and tag_end_<tagname>, that will be called when 
    the <tagname> and </tagname> XML idenifiers are encountered, respectively.
    
    The advantage of using 'tag_<tagname>' is that it's intent is obvious and
    all of the information related to the tag is available when the method
    is called. The advantage to the begin/end pairs is that when nested tags
    are used, it can be helpful to preform some preliminary actions in the
    'begin_tag_<tagname>' call to set the stage for further processing.
    The corresponding 'end_tag_<tagname>' is called with exactly the same 
    symantics as 'tag_<tagname>'. The only difference is the name of the
    method.
    
    If a subclass would like to use the built-in validation capabilities of
    Essax, they should use the 'register_tag_format' and 
    'register_tag_attributes' methods prior to calling the 'parse' method.
    Validation is completely optional. If the format of a tag is 
    registered with the Essax base class, all occurances of that tag will
    be validated against the specified format. Occurances of tags for which
    no format has been registered will not be validated. Additionally,
    the attributes of a tag are checked independently of the tag
    format. This allows user code to take advantage of attribute checking
    and type conversion without requiring a specific tag contents.

    It should be noted that subclasses of an arbitrarily deep inheritance
    tree can use the two registration methods to override the values set by 
    their baseclasses. In this way, a subclass can modify the legal format of
    the XML document. If this is done, care should be taken to insure that
    the format changes do not break constraints that the base classes are
    relying on to be true.
    
    A simple call to 'parse' will initiate the parsing process. If validation
    is being used and, at any point, a tag or attribute deviates from the
    specified format, a ValidationError exception will be raised. The exception
    will contain a description of the error, the file name, and line number 
    at which the error occured. 
    
    When a tag handler method is invoked, the parser's 'this_tag' attribute
    will always refer to the current tag being processed. This will be an
    instance of GenericTag and will contain all of the information available
    for the current tag. As 'begin_tag_<tagname>' calls occur before 
    any of the tag's contents have been processed, that information will,
    of course, be lacking. During the 'tag_<tagname>' and 'tag_end_<tagname>'
    calls, however, all information is available.
    
    A distinct difference from normal SAX parsers, is in the contents of the
    attributes dictionary that is stored in the GenericTag instances.
    By default, all values in the dictionary are strings (the normal SAX
    behaviour). If, however, a 'register_tag_attributes' call was used to
    specify the types of the attributes, the values in the attributes    
    dictionary will objects of the specified type. For example,
    if an attribute were specified as 'number_of_elements=integer'
    the corresponding value in the attributes dicionary would be a Python
    integer, not a string. Subclasses can provide their own attribute types
    simply by defining a method named 'convert_attr_<typename>'
    that accepts the contents of an attribute string and returns the converted
    value. 
    """
    
    def get_current_tag(self):
        if len(self.tag_stack) == 0:
            return None
        return self.tag_stack[-1]

        
    this_tag = property(get_current_tag, None, None)
    
    

    def __init__(self, file_name, err_unknown_tags=False):
        XMLFilterBase.__init__(self)
        
        self.file_name     = file_name
        self.locator       = None
        self.tag_stack     = deque()
        
        self._keep_subtags = dict()
        self._attr_conv    = dict()  # tag_names => (required_attrs, opt_attrs)
        self._validators   = dict()
        
        self._err_unknown_tag = err_unknown_tags
                    
        

    def keep_subtags_for(self, tag_names):
        """ By default, the GenericElements used to hold tag information are
        discarded when the processing for the tag is complete. This method
        instructs the parser to retain the GenericElements for all tags
        encountered within the specified tags. If kept, the GenericElement
        instances for the subtags will be appended to the 'sub_tags' variable
        of the GenericElement corresponding to a tag specified in the argument.
        
        The 'tag_names' argument is a whitespace separated string containing
        the names of the tags for which subtags information should be 
        retained."""
        for t in tag_names.split():
            self._keep_subtags[ t ] = True
        
        
        
    def register_tag_format(self, tag_name, format):
        """ This method allows the format/layout of a tag to be explicitly
        specified. If this method is used, all occurances of the specified tag
        in an XML document will be checked for conformance with the provided
        format. If the parser encounters something outside the bounds of the
        format, a ValidationError exception will be raised. It should be
        noted that these validation checks are preformed as part of the
        parsing process and the exception can be generated at any time
        (which immediately aborts the parsing of the XML document).
        
        The 'format' parameter is a white-space separated string of tag names
        that have an optional count specifier and that may be grouped in () to
        indicate that the order of the tags is not important (by default, the
        order of the tags is enforced). While the format may look similar to
        a DTD specifier, they do not share the same symantics. Most notably,
        a closing ) may NOT be followed by a ?, +, or *
        
        Format description:
                
            Tags are order dependent by default
            Tags grouped in ( ) are not order dependent
            
            Optional Count Specifiers for tags:
                ? for 0 or 1
                + for 1 or more
                * for 0 or more
                
            Unlike a DTD, no count specifier is allowed at the end of a (...)

        Example: 'fname midname? lname (age sex) (phone* address+) vehicle*'
        """
        self._validators[ tag_name ] = ValidatorCache( format )
                
        
        
    def register_tag_attributes(self, tag_name, required, optional=None):
        """ This method allows the set of required and optional attributes for
        a tag to be specified. Additionally, the type of data stored in the
        attribute may be indicated. If used, run-time checks will be preformed
        to ensure that all required attributes are present and that the types 
        (if specified) are correct. The 'required' and 'optional' arguments         
        accept strings containing a comma-separated list of attribute names.
        By default, all attributes are strings. However, the type of an 
        attribute may be specified by following the attribute name with a 
        '= type_name'. If this is done, a conversion method will be
        automatically called to convert the attribute string into a Python
        object of the appropriate type. EssaxParser has built-in support for
        integer, float, boolean, and string (the default). Subclasses may 
        define their own types simply by definging a method named
        'convert_attr_<type_name>' that accepts a string containing attribute
        text and returns a Python object of the appropriate type.
        
        Example:
            rta = self.register_tag_attributes
            rta('atom', 'name, atomic_number=integer', 'weight=float')
        """
        req      = True
        req_dict = dict()
        opt_dict = dict()    
        atstr    = required
    
        if not optional is None:
            atstr = required + ',#,' + optional
    
        for c in atstr.split(','):
            lst = [ part for part in c.split('=') if part ]
            
            if lst[0] == '#':
                req = False
                continue
                                
            attr_name = lst[0].strip()
            
            if len(lst) == 1:
                attr_type = 'string'
            else:
                attr_type = lst[1].strip()
        
            conversion_func = getattr(self, 'convert_attr_' + attr_type, None)
        
            if conversion_func is None:
                tpl = (self.__class__.__name__, attr_type)
                raise Exception('class %s is missing attr_convert_%s method \
for attribute type conversion' % tpl)
        
            if req:
                req_dict[attr_name] = conversion_func
            else:
                opt_dict[attr_name] = conversion_func
        
        self._attr_conv[ tag_name ] = (req_dict, opt_dict)

        
        
        
    def _convert_attrs(self, tag_name, attrs):    
        
        if not tag_name in self._attr_conv:
            #
            # The SAX API does not guarantee that the attrs dictionary will
            # persist between start/end element callbacks. Return a copy
            # of the attributes to be on the safe side.
            #
            return dict(attrs) 
        
        new_attrs = dict()
        req, opt  = self._attr_conv[ tag_name ]
        
        for name, value in attrs.items():
            if name in req:                    
                new_attrs[ name ] = req[name](value)
            elif name in opt:
                new_attrs[ name ] = opt[name](value)
            else:
                new_attrs[ name ] = value
                    
        for name in req.iterkeys():
            if not name in attrs:                    
                raise MissingRequiredAttribute(tag_name, name, self)
                
        return new_attrs
        

    
    def convert_attr_string(self, value):
        return value # for completeness
    
    
    
    def convert_attr_integer(self, value):
        try:
            return int(value)
        except ValueError:
            raise AttributeConversionError(self, 
                  '"%s" is not an integer' % value)
        
    
        
    def convert_attr_float(self, value):
        try:
            return float(value)
        except ValueError:
            raise AttributeConversionError(self,
                  '"%s" is not a floating point number' % value)
        
                  
                  
    def convert_attr_boolean(self, value):
        """ Acceptable values for True are case-insensitive compares against:
                '1', 'true',  'yes', and 'enabled'
            For False, case insensitive compares agains:
                '0', 'false', 'no', and 'disabled'
        """
        lv = value.lower()
        if   lv in ('1', 'true',  'yes', 'enabled'):
            return True
        elif lv in ('0', 'false', 'no',  'disabled'):
            return False
        else:
            raise AttributeConversionError(self,
                  '"%s" is not a boolean' % value)
    
                
                
    def parse(self):
        """ This method runs the SAX parser and causes the user-defined tag
        handlers to be invoked. If any errors are encountered. An exception
        will be thrown
        """
        parser = xml.sax.make_parser()
    
        parser.setContentHandler (          self          )
        parser.setErrorHandler   ( xml.sax.ErrorHandler() )
        
        parser.setFeature( xml.sax.handler.feature_validation  , None )
        parser.setFeature( xml.sax.handler.feature_external_ges, None )
        parser.setFeature( xml.sax.handler.feature_external_pes, None )

        try:
            parser.parse( self.file_name )
        except RuntimeParseError, e:
            print '-'*80
            print 'Parsing Failed:'
            print '%s:%s %s' % (e.file_name, e.line_number, str(e))
            print '-'*80
            raise
        
                            
        
    def characters(self, contents):
        self.this_tag._text.append( contents )
                            	
                
                        
    def setDocumentLocator(self, locator):
        self.locator = locator
        

        
    def exec_tag_handler(self, tag_handler):
        
        if tag_handler is None:
            return
        
        try:
            
            tag_handler()
            
        except RuntimeParseError:
            raise
            
        except Exception, e:
            #
            # This should give the calling routine a way to discover
            # where the parser was in the file when the exception occured
            #
            e.essax_exception = RuntimeParseError(self)
            raise
        
        
        
    def startElement(self, tag_name, attrs):

        vtag = self.this_tag
        
        self.tag_stack.append(GenericTag(tag_name, 
                                         self.locator.getLineNumber()))
        
        if vtag and not vtag._validator is None:
            top_tag = len(self.tag_stack) == 1
            try:
                ret = vtag._validator.validate_tag( tag_name )
                if ret == V_CHECK_NEXT:
                    raise ValidationError('Tag "%s" is not allowed here' 
                                          % tag_name)
            except ValidationError, e:
                e.set_parser(self)
                raise
                                        
                                             
        if tag_name in self._validators:
            tag = self.this_tag
            tag._validator = self._validators[ tag_name ].pop_validator()
            
        if tag_name in self._keep_subtags:
            self.this_tag.keep_subtags = True
                    
        self.this_tag.attrs = self._convert_attrs( tag_name, attrs )
                
        if not hasattr(self, 'tag_' + tag_name):
            
            tag_handler = getattr(self, 'tag_begin_' + tag_name, None)
            
            if tag_handler is None or not hasattr(self, 'tag_end_' + tag_name):
                if self._err_unknown_tag:
                    raise UnknownTag(tag_name, self)                
                else:
                    print '!! Remove This !! Warning! Unknown Tag:', tag_name

            self.exec_tag_handler( tag_handler )
    
	    	
	
    def endElement(self, tag_name):                
        
        if not self.this_tag._validator is None:                        
            v = self.this_tag._validator
            
            if not v.is_satisfied():
                #
                # Could be exiting early is ok. Try an arbitrary tag to see
                #
                if not v.validate_tag('__end_tag__') == V_CHECK_NEXT:
                                       
                    e = ValidationError('The format of tag "%s" is incorrect' %
                        tag_name)
                    e.set_parser(self)
                    raise e
            
            self._validators[tag_name].push_validator( v )
            self.this_tag._validator = None
	
        tag_handler = getattr(self, 'tag_' + tag_name, None)
        
        if tag_handler is None:
            tag_handler = getattr(self, 'tag_end_' + tag_name, None)
            
        self.exec_tag_handler( tag_handler )        
        
        sub_tag = self.tag_stack.pop()
        
        if self.this_tag and self.this_tag.keep_subtags:
            self.this_tag.sub_tags.append( sub_tag )
        
                    
                
                
    def error( ex ):        
        ex.essax_exception = RuntimeParseError(self)
	raise ex
    
    def fatalError( ex ):
        ex.essax_exception = RuntimeParseError(self)
	raise ex
    
    def warning( ex ):
        ex.essax_exception = RuntimeParseError(self)
	raise ex

    
    
###############################################################################
#
#                             Validation
#
##############################################################################

# example = '(descrip? author?) (age sex) fixed* area (fixed* area*)'
#
#
# default: Order dependent
# group in ( ) to make order independent
# ? for 0 or 1
# + for 1 or more
# * for 0 or more
# Unlike DTD, no specifier is allowed at the end of a (...)
#
#




class VElement (object):
    
    def __init__(self, tag_name, count_modifier):
        self.tag_name   = tag_name
        self.count_mod  = count_modifier
        
                
        
class VUnOrderedList (list):
    pass
        
        
        
        
class ElementValidator (object):
    
    def __init__(self, element):
        self.element = element
        self.reset()
        
        
    def reset(self):
        self.count      = 0
        self._satisfied = False
        
        if self.element.count_mod in ('?', '*'):
            self._satisfied = True
                    
    def is_satisfied(self):
        return self._satisfied
            
            
    def validate_tag(self, tag_name):
        
        if not tag_name == self.element.tag_name:
            return V_CHECK_NEXT
        
        if self.count == 1 and self.element.count_mod in ('?', ''):
            raise ValidationError('Only one "%s" tag is allowed' %
                                   self.element.tag_name)
        
        self.count += 1
        self._satisfied = True
        
        if self.element.count_mod in ('?', ''):
            return V_DONE
        else:
            return V_OK
        
        
        
    def __str__(self):
        return self.element.tag_name + self.element.count_mod
        
        
                
class UnOrderedListValidator (object):
    
    def __init__(self, velement_list):
        self.edict = dict()
        for element in velement_list:
            self.edict[ element.tag_name ] = ElementValidator(element)
            
            
    def reset(self):
        for ev in self.edict.itervalues():
            ev.reset()

            
    def is_satisfied(self):
        for ev in self.edict.itervalues():
            if not ev.is_satisfied():
                return False
        return True
            
            
    def validate_tag(self, tag_name):        
        if tag_name in self.edict:
            self.edict[ tag_name ].validate_tag(tag_name)
            return V_OK
        else:
            if not self.is_satisfied():
                raise ValidationError('Invalid location for tag "%s"' %
                                      tag_name)

            return V_CHECK_NEXT
        
    def __str__(self):
        return '(' + ' '.join([str(e) for e in self.edict.iteritems()]) + ')'
        


    
class OrderedListValidator (object):
    
    def __init__(self, tag_format):
        
        elements = parse_tag_format( tag_format )
        
        self.elist = list()
        for e in elements:
            if isinstance(e, VElement):
                self.elist.append( ElementValidator(e) )
            else:
                self.elist.append( UnOrderedListValidator(e) )
                
        self.eiter           = iter( self.elist )
        self.next_element    = self.eiter.next()
        self.is_last_element = False
        
        
        
    def reset(self):
        for e in self.elist:
            e.reset()
            
        self.eiter         = iter( self.elist )
        self.next_element  = self.eiter.next()
        self.is_last_element = False

        
    def is_satisfied(self):
        return self.is_last_element and self.next_element.is_satisfied()
        
        
    def validate_tag(self, tag_name):
        ret = self.next_element.validate_tag( tag_name )
        
        if ret == V_OK:
            return V_OK
        
        try:
            self.next_element = self.eiter.next()
        except StopIteration:
            self.is_last_element = True
            return ret # pass along the V_DONE or V_CHECK_NEXT
        
        if ret == V_DONE:
            return V_OK
        #
        # ret is V_CHECK_NEXT... so... check the next one
        #        
        ret = self.next_element.validate_tag( tag_name )
        
        while ret == V_CHECK_NEXT and self.next_element.is_satisfied():
            try:
                self.next_element = self.eiter.next()
            except StopIteration:
                self.is_last_element = True
                return V_DONE
            
            ret = self.next_element.validate_tag( tag_name )
        
        if ret == V_OK:
            return V_OK
        
        if ret == V_DONE:
            try:
                self.next_element = self.eiter.next()
            except StopIteration:
                self.is_last_element = True
                return V_DONE
            else:
                return V_OK
            
        # two V_CHECK_NEXT in a row means it's invalid
        tpl = (tag_name, self.next_element)
        raise ValidationError('Invalid location for tag "%s". Expected "%s"' %
                              tpl)
    
    
    def __str__(self):
        ' '.join([str(e) for e in self.elist])
        

        

        
class ValidatorCache (object):
    
    def __init__(self, tag_format):
        self._format     = tag_format
        self._validators = deque()        
        
        
    def pop_validator(self):
        if self._validators:
            return self._validators.pop()
        else:
            return OrderedListValidator( self._format )
    
        
    def push_validator(self, v):
        self._validators.append( v )
        
        
        
def parse_tag_format( tformat ):
    
    elements    = list()
    in_brace    = False
    current_uol = None
    
    for e in tformat.split():
        tstart    = 0
        tend      = 1
        count_mod = ''
        
        
        
        if e[0] == '(':
            if in_brace:
                raise InvalidTagFormat('Nested "()" are not \
supported in validation declarations: "%s"'%tformat)
            else:
                in_brace    = True
                current_uol = VUnOrderedList()
            tstart = 1
            
            
        if e[-1] == ')': 
            tend += 1
                                
        if e[len(e) - tend] in ('*', '?', '+'):
            count_mod = e[len(e) - tend]
            tend += 1
            
        last_idx = len(e) - tend + 1
            
        #t = in_brace and 'unordered' or 'normal   '
        #print 'Creating new %s element (%s, %s)' % (t, e[tstart:last_idx],
        #                                            count_mod)
        
        new_element = VElement( e[tstart:last_idx], count_mod )
        
        if in_brace:
            current_uol.append( new_element )
        else:
            elements.append( new_element )
            
            
        if e[-1] == ')':
            if not in_brace:
                raise InvalidTagFormat('Unmatched ")" in \
validation declaration: "%s"'%tformat)
            else:
                elements.append( current_uol )
                in_brace    = False                
                current_uol = None
                
    if in_brace:
        raise InvalidTagFormat(
              'Unmatched "(" in validation declaration: "%s"' % tformat)
    return elements
            
        
            
#------------------------------------------------------------------------------
#                        Validation Test code                                 -
#------------------------------------------------------------------------------

if __name__ == '__main__':
    
    def print_element(element, indent):
        print indent, '(%s, %s)' % (element.tag_name, element.count_mod)

    def test_parse( decl ):
        l = parse_tag_format( decl )
        print '-'*80
        print decl
        print ''
        for e in l:
            if isinstance(e, list):
                for x in e:
                    print_element(x, '    ')
            else:
                print_element(e, '')

    
    #test_parse( attrs )
    #test_parse( '( woo hoo shmArea*  ) ( sall (good blah) )' )
    
    #decl = 'first_tag (descrip? author?) (male female) fixedAddr* shmArea (fixedAddr* shmArea*)  import*'
    
    #seq = 'first_tag author female male fixedAddr fixedAddr shmArea shmArea shmArea import import blah'
    

    def test():
        #decl = "(one two three) lots+ theend"
        #seq  = "three one two"
        decl = "alpha beta? zeta"
        seq  = "zeta beta"
        
        olv = OrderedListValidator( decl )
        
        rmap = dict()
        rmap[ V_DONE ] = "V_DONE"
        rmap[ V_OK   ] = "V_OK"
        rmap[ V_CHECK_NEXT ] = "V_CHECK_NEXT"
        
        print '-'*80
        print 'Valid: "%s"' % decl
        print ''
        print 'Trying: "%s"' % seq
        print ''
        for s in seq.split():
            print 'Checking %s' % s
            print '    ', rmap[ olv.validate_tag( s ) ]
            
        print 'Satisfied: ', olv.is_satisfied()
        
    test()
